<?php

namespace App\Http\Controllers;

use App\Http\Requests\User\DeleteUserRequest;
use App\Http\Requests\User\SignInUserRequest;
use App\Http\Requests\User\SignUpUserRequest;
use App\Http\Requests\User\UpdateUserRequest;
use App\Models\User;
use App\Traits\HttpResponses;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    use HttpResponses;

    /**
     * Generate an access token.
     */
    private function generateToken($user, $email, $validUntil = null): array
    {
        if (!$validUntil) $validUntil = now()->addWeek();

        $token = $user->createToken($email, ['*'], $validUntil)->plainTextToken;

        return [
            'access_string' => $token,
            'valid_until' => $validUntil
        ];
    }

    /**
     * Store a newly created user.
     */
    public function signUp(SignUpUserRequest $request)
    {
        $validatedData = $request->validated();

        $user = User::create($validatedData);

        return $this->success(
            'Account created successfully.',
            [
                'user' => $user,
                'token' => $this->generateToken($user, $validatedData['email'])
            ]
        );
    }

    /**
     * Authenticate a user.
     */
    public function signIn(SignInUserRequest $request)
    {
        $validatedData = $request->validated();

        if (!Auth::attempt($validatedData)) {
            return $this->failure(
                'Incorrect email or password.',
                401
            );
        }

        $user = Auth::user();

        return $this->success(
            'Account was signed in successfully.',
            [
                'user' => $user,
                'token' => $this->generateToken($user, $validatedData['email'])
            ]
        );
    }

    /**
     * Deauthenticate a user.
     */
    public function signOut()
    {
        $user = Auth::user();

        $user->currentAccessToken()->delete();

        return $this->success('Your have been successfully signed out.');
    }

    /**
     * Update a user.
     */
    public function update(UpdateUserRequest $request)
    {
        $validatedData = $request->validated();

        $user = Auth::user();

        $user->update($validatedData);

        return response()->json([
            'success' => true,
            'message' => 'Your account was updated successfully.',
            'data' => [
                'user' => $user
            ]
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(DeleteUserRequest $request)
    {
        $user = Auth::user();

        $validatedPassword = $request->validated()['password'];

        if (!Hash::check($validatedPassword, $user->getAuthPassword())) {
            return $this->failure(
                'The password you entered could not be verified. Please try again.',
                401
            );
        }

        $user->delete();
        $user->tokens()->delete();

        return $this->success('Your account has been permanently closed.');
    }
}
