/*autors: Veronika Osko
  datums: 10.04.2025.
  Jelgavas tehnikums
  110.grupa*/

using System;  // Ievadām nepieciešamo bibliotēku, kas satur pamata funkcijas, piemēram, Console.
using System.Collections.Generic;  // Ievadām kolekciju funkcionalitāti, lai izmantotu Listu.
using System.IO;  // Ievadām I/O funkcionalitāti, lai varētu strādāt ar failiem, ja nepieciešams.

class FileSystemObject  // Abstrakta klase, kas pārstāv jebkuru failu sistēmas objektu, gan mapes, gan datnes.
{
    public string Name { get; set; }  // Objekta nosaukums.
    public Folder Parent { get; set; }  // Vecāks objekts, kas var būt mape, kas satur šo objektu.

    public virtual void Show() { }  // Virtuālā metode, kas tiek pārmantota, lai parādītu objektu saturu (mapēm un datnēm).
}

class Folder : FileSystemObject  // Mape, kas ir FileSystemObject apakšklase.
{
    public List<FileSystemObject> Children = new List<FileSystemObject>();  // Saraksts, kas satur mapes bērnus (mājas vai datnes).

    public override void Show()  // Pārmantojam `Show()` metodi, lai parādītu mapes bērnus.
    {
        foreach (var child in Children)  // Izstaigājam visus bērnus, kas ir mapē.
        {
            // Ja bērns ir mape, izvadām to ar [] (pārbaudām ar `is Folder`).
            // Ja bērns ir datne, izvadām to ar ~[] (pārbaudām ar `is File`).
            Console.WriteLine(child is Folder ? $"/[{child.Name}]" : $"/~[{child.Name}]~");
        }
    }

    public void AddChild(FileSystemObject child)  // Metode bērnu pievienošanai.
    {
        // Pārbaudām, vai bērns jau eksistē, lai izvairītos no dubultiem nosaukumiem.
        if (!Children.Exists(c => c.Name == child.Name)) 
            Children.Add(child);  // Ja tāda nav, pievienojam bērnu.
    }

    public void RemoveChild(string name)  // Metode bērna dzēšanai pēc nosaukuma.
    {
        var child = Children.Find(c => c.Name == name);  // Meklējam bērnu ar konkrētu nosaukumu.
        if (child != null)  // Ja bērns ir atrasts, dzēšam to.
            Children.Remove(child);
    }
}

class File : FileSystemObject  // Datnes klase, kas arī ir `FileSystemObject` apakšklase.
{
    public void Edit()  // Metode faila rediģēšanai (šobrīd tikai izdrukā ziņojumu).
    {
        Console.WriteLine($"Editing file {Name}.txt");  // Izdrukā ziņojumu par faila rediģēšanu.
    }
}

class FileSystem  // Failu sistēmas galvenā klase, kas simulē darbības ar failiem un mapēm.
{
    static Folder root = new Folder { Name = "C:" };  // Sākotnējā mape, kas atbilst c: diskam.
    static Folder current = root;  // Pašreizējā aktīvā mape, sākot ar sakni.

    static void Main()  // Galvenā programma, kas darbojas kā komandrindas interfeiss.
    {
        while (true)  // Bezgalīga cilpa, kas pieļauj lietotāja ievadi un izpilda komandas.
        {
            Console.Write($"{current.Name}/> ");  // Parāda pašreizējo atrašanās vietu uz ekrāna.
            string command = Console.ReadLine().Trim();  // Lasām komandu no lietotāja.
            var args = command.Split(' ', StringSplitOptions.RemoveEmptyEntries);  // Ievadi sadalām pa argumentiem.

            if (args.Length == 0) continue;  // Ja ievade ir tukša, turpinām nākamo iterāciju.

            switch (args[0].ToLower())  // Pārbaudām pirmo komandu argumentu, kas norāda darbību.
            {
                case "mkdir":  // Ja komanda ir "mkdir" — izveidot mapi.
                    if (args.Length > 1)  // Ja ievadīti ir papildu argumenti.
                        current.AddChild(new Folder { Name = args[1], Parent = current });  // Pievienojam jaunu mapi.
                    break;

                case "create":  // Ja komanda ir "create" — izveidot datni.
                    if (args.Length > 1)  // Ja ievadīti ir papildu argumenti.
                        current.AddChild(new File { Name = args[1], Parent = current });  // Pievienojam jaunu datni.
                    break;

                case "dir":  // Ja komanda ir "dir" — parādīt mapes bērnus.
                    current.Show();  // Izsaucam `Show()` metodi, kas parāda bērnus.
                    break;

                case "cd":  // Ja komanda ir "cd" — mainīt aktuālo mapi.
                    if (args[1] == ".." && current.Parent != null)  // Ja args[1] ir "..", ejam atpakaļ.
                        current = current.Parent;  // Pārvietojamies uz vecāku mapi.
                    else  // Pretējā gadījumā, pārvietojamies uz bērnu mapi.
                        current = current.Children.Find(c => c.Name == args[1]) as Folder;  
                    break;

                case "rm":  // Ja komanda ir "rm" — dzēst bērnu (mapi vai datni).
                    current.RemoveChild(args[1]);  // Meklējam bērnu pēc nosaukuma un dzēšam to.
                    break;

                case "edit":  // Ja komanda ir "edit" — rediģēt datni.
                    var file = current.Children.Find(c => c.Name == args[1]) as File;  // Atrodam datni.
                    file?.Edit();  // Ja datne atrasta, izsaucam tās `Edit()` metodi.
                    break;

                case "exit":  // Ja komanda ir "exit" — iziet no programmas.
                    return;  // Aizver galveno metodi, kas izbeidz programmu.
            }
        }
    }
}

using System;
using System.Collections.Generic; // Nepieciesams List<string>
using System.Linq; // Nepieciesams Enumerable.Range un .ToArray()

// Klase, kas reprezentē bankas kontu
public class BankasKonts
{
    // Privatas ipasibas, lai tam nevaretu tiesi pieklut arpus klases
    private string kontaNumurs;
    private string kontaIpasniekaVards;
    private decimal atlikums;

    // Static lauks, lai nodrosinatu unikalu kontu numerus
    private static int nakosaisKontaNumurs = 1000; // Sāksies ar 1000

    // Konstruktors, kas inicialize konta ipasibas
    public BankasKonts(string ipasniekaVards, decimal sakumaAtlikums)
    {
        // Genere unikalu konta numeru
        this.kontaNumurs = (++nakosaisKontaNumurs).ToString(); // Palielina un parveido par stringu
        this.kontaIpasniekaVards = ipasniekaVards;
        
        // Nodrosina, ka sakuma atlikums nav negativs
        if (sakumaAtlikums < 0)
        {
            // Izmet iznemumu, ja sakuma atlikums ir nederigs
            throw new ArgumentException("Sakuma atlikums nevar but negativs.");
        }
        this.atlikums = sakumaAtlikums;

        // Izvada zinojumu par konta izveidi
        Console.WriteLine($"[INFO] Jauns konts '{kontaNumurs}' izveidots prieks '{kontaIpasniekaVards}'.");
    }

    // Destruktors (finalizer) - tiek izsaukts, kad objekts tiek savakts ar atkritumu savaceju
    ~BankasKonts()
    {
        // Registre konta objekta iznicinasanu
        Console.WriteLine($"[INFO] Konts '{kontaNumurs}' tiek iznicinats.");
    }

    // Publikas ipasibas (getter), lai pieklutu konta numeram un ipasnieka vardam
    public string KontaNumurs
    {
        get { return kontaNumurs; }
    }

    public string KontaIpasniekaVards
    {
        get { return kontaIpasniekaVards; }
    }

    // Metode naudas noguldisanai
    public void Noguldit(decimal summa)
    {
        // Parbauda, vai summa ir pozitiva
        if (summa <= 0)
        {
            // Izmet iznemumu, ja summa ir nederiga
            throw new ArgumentException("Noguldama summa nedrikst but negativa vai nulle.");
        }
        atlikums += summa; // Pieskaita summu atlikumam
        Console.WriteLine($"Noguldits {summa:C}. Jauns atlikums: {atlikums:C}");
    }

    // Metode naudas iznemsanai
    public void Izņemt(decimal summa)
    {
        // Parbauda, vai summa ir pozitiva
        if (summa <= 0)
        {
            // Izmet iznemumu, ja summa ir nederiga
            throw new ArgumentException("Iznemama summa nedrikst but negativa vai nulle.");
        }
        // Parbauda, vai konta ir pietiekami daudz naudas
        if (summa > atlikums)
        {
            // Izmet iznemumu, ja nav pietiekami daudz lidzeklu
            throw new InvalidOperationException("Nepietiek lidzeklu konta.");
        }
        atlikums -= summa; // Aten summu no atlikuma
        Console.WriteLine($"Iznemts {summa:C}. Jauns atlikums: {atlikums:C}");
    }

    // Metode konta informacijas paradisanai
    public void ParaditKontaInformaciju()
    {
        Console.WriteLine("\n--- Konta Informacija ---");
        Console.WriteLine($"Konta numurs: {kontaNumurs}");
        Console.WriteLine($"Konta ipasnieks: {kontaIpasniekaVards}");
        Console.WriteLine($"Atlikums: {atlikums:C}"); // Formats valutas attelosanai
        Console.WriteLine("-------------------------\n");
    }
}

// Galvena programma
public class BankasSistema
{
    // Saraksts, kura glabajas visi bankas konti
    private static List<BankasKonts> konti = new List<BankasKonts>();

    public static void Main(string[] args)
    {
        int izvele; // Lietotaja izvele
        do
        {
            // Parada galveno izvelni
            ParaditIzvelni();
            // Nolasas lietotaja ievadi
            if (int.TryParse(Console.ReadLine(), out izvele))
            {
                // Apstrada lietotaja izveli
                switch (izvele)
                {
                    case 1:
                        IzveidotJaunuKontu();
                        break;
                    case 2:
                        NogulditNauduKonta();
                        break;
                    case 3:
                        IznemtNauduNoKonta();
                        break;
                    case 4:
                        ParaditKontaInformaciju();
                        break;
                    case 5:
                        DzestKontu();
                        break;
                    case 6:
                        // Iziet no sistemas
                        Console.WriteLine("Iziet no sistemas. Uz redzesanos!");
                        break;
                    default:
                        // Kļudaina ievade
                        Console.WriteLine("Nederiga izvele. Ludzu, meginat velreiz.");
                        break;
                }
            }
            else
            {
                // Kļudaina ievade
                Console.WriteLine("Nederiga ievade. Ludzu, ievadiet skaitli.");
            }
            Console.WriteLine(); // Tuksa rinda estetikai
        } while (izvele != 6); // Turpina darbu, kamēr lietotajs neizvelas iziet
    }

    // Metode, kas parada galveno izvelni
    private static void ParaditIzvelni()
    {
        Console.WriteLine("Ludzu, izvelieties darbibu:");
        Console.WriteLine("1. Izveidot jaunu bankas kontu");
        Console.WriteLine("2. Noguldit naudu konta");
        Console.WriteLine("3. Iznemt naudu no konta");
        Console.WriteLine("4. Paradit konta informaciju");
        Console.WriteLine("5. Dzest kontu");
        Console.WriteLine("6. Iziet no sistemas");
        Console.Write("Jusu izvele: ");
    }

    // Metode jauna bankas konta izveidei
    private static void IzveidotJaunuKontu()
    {
        Console.Write("Ievadiet konta ipasnieka vardu: ");
        string vards = Console.ReadLine();

        decimal sakumaAtlikums = 0;
        bool validIevade = false;
        while (!validIevade)
        {
            Console.Write("Ievadiet sakuma atlikumu: ");
            // Megina parse ievadi un parbauda vai ir derigs skaitlis
            if (decimal.TryParse(Console.ReadLine(), out sakumaAtlikums) && sakumaAtlikums >= 0)
            {
                validIevade = true;
            }
            else
            {
                // Zino par nederigu ievadi
                Console.WriteLine("Nederigs atlikums. Ludzu, ievadiet pozitivu skaitli vai nulli.");
            }
        }

        try
        {
            // Izveido jaunu BankasKonts objektu
            BankasKonts jaunsKonts = new BankasKonts(vards, sakumaAtlikums);
            // Pievieno jauno kontu sarakstam
            konti.Add(jaunsKonts);
            Console.WriteLine("Konts veiksmigi izveidots!");
        }
        catch (ArgumentException ex)
        {
            // Apstrada iznemumus, kas saistiti ar argumentiem
            Console.WriteLine($"Kļuda: {ex.Message}");
        }
    }

    // Metode naudas noguldisanai
    private static void NogulditNauduKonta()
    {
        Console.Write("Ievadiet konta numeru: ");
        string kontaNumurs = Console.ReadLine();

        // Atrod kontu pec numura
        BankasKonts konts = konti.Find(k => k.KontaNumurs == kontaNumurs);

        // Parbauda, vai konts tika atrasts
        if (konts == null)
        {
            Console.WriteLine("Konts ar sadu numeru nav atrasts.");
            return;
        }

        decimal summa;
        Console.Write("Ievadiet noguldamo summu: ");
        // Megina parse summu
        if (decimal.TryParse(Console.ReadLine(), out summa))
        {
            try
            {
                // Izsauc noguldisanas metodi
                konts.Noguldit(summa);
            }
            catch (ArgumentException ex)
            {
                // Apstrada iznemumus par nederigu summu
                Console.WriteLine($"Kļuda: {ex.Message}");
            }
        }
        else
        {
            // Zino par nederigu ievadi
            Console.WriteLine("Nederiga summa. Ludzu, ievadiet skaitli.");
        }
    }

    // Metode naudas iznemsanai
    private static void IznemtNauduNoKonta()
    {
        Console.Write("Ievadiet konta numeru: ");
        string kontaNumurs = Console.ReadLine();

        // Atrod kontu pec numura
        BankasKonts konts = konti.Find(k => k.KontaNumurs == kontaNumurs);

        // Parbauda, vai konts tika atrasts
        if (konts == null)
        {
            Console.WriteLine("Konts ar sadu numeru nav atrasts.");
            return;
        }

        decimal summa;
        Console.Write("Ievadiet iznemamo summu: ");
        // Megina parse summu
        if (decimal.TryParse(Console.ReadLine(), out summa))
        {
            try
            {
                // Izsauc iznemsanas metodi
                konts.Izņemt(summa);
            }
            catch (ArgumentException ex)
            {
                // Apstrada iznemumus par nederigu summu
                Console.WriteLine($"Kļuda: {ex.Message}");
            }
            catch (InvalidOperationException ex)
            {
                // Apstrada iznemumus par nepietiekamiem lidzekliem
                Console.WriteLine($"Kļuda: {ex.Message}");
            }
        }
        else
        {
            // Zino par nederigu ievadi
            Console.WriteLine("Nederiga summa. Ludzu, ievadiet skaitli.");
        }
    }

    // Metode konta informacijas paradisanai
    private static void ParaditKontaInformaciju()
    {
        Console.Write("Ievadiet konta numeru: ");
        string kontaNumurs = Console.ReadLine();

        // Atrod kontu pec numura
        BankasKonts konts = konti.Find(k => k.KontaNumurs == kontaNumurs);

        // Parbauda, vai konts tika atrasts
        if (konts == null)
        {
            Console.WriteLine("Konts ar sadu numeru nav atrasts.");
            return;
        }
        // Izsauc konta informacijas paradisanas metodi
        konts.ParaditKontaInformaciju();
    }

    // Metode konta dzesanai
    private static void DzestKontu()
    {
        Console.Write("Ievadiet konta numeru, ko dzest: ");
        string kontaNumurs = Console.ReadLine();

        // Atrod kontu pec numura
        BankasKonts kontsDzēšanai = konti.Find(k => k.KontaNumurs == kontaNumurs);

        // Parbauda, vai konts tika atrasts
        if (kontsDzēšanai == null)
        {
            Console.WriteLine("Konts ar sadu numeru nav atrasts.");
            return;
        }

        // Nņem kontu no saraksta
        konti.Remove(kontsDzēšanai);
        // Izsauc Garbage Collector, lai iespejams izsauktu destruktoru (nav garantets)
        GC.Collect();
        GC.WaitForPendingFinalizers(); // Gaida, kamēr destruktori pabeidz darbu
        Console.WriteLine($"Konts '{kontaNumurs}' veiksmigi dzests.");
    }
}