﻿namespace ConsoleApp2 {
    // Enumerācija, kas definē faila tipu - mape vai datne
    internal enum FailaTips {
        MAPE, // Norāda, ka fails ir mape
        DATNE // Norāda, ka fails ir datne
    }

    // Klase, kas reprezentē failu sistēmas objektu (mapi vai datni)
    internal class Fails {
        public readonly string nosaukums; // Faila vai mapes nosaukums
        public Fails vecaks; // Norāde uz vecāku mapi
        public readonly List<Fails> berni = new(); // Saraksts ar bērniem Fails
        private readonly FailaTips tips; // Faila tips (MAPE vai DATNE)

        // Konstruktors, kas inicializē faila nosaukumu un tipu
        public Fails(string nosaukums, FailaTips tips) {
            this.nosaukums = nosaukums;
            this.tips = tips;
        }

        // Pārbauda, vai fails ir datne
        public bool irDatne() {
            return tips == FailaTips.DATNE;
        }

        // Pārbauda, vai fails ir mape
        public bool irMape() {
            return tips == FailaTips.MAPE;
        }

        // Izvada visus bērnus zem šī faila (mapes)
        public void paradit() {
            if (berni.Count == 0) { // Ja nav bērnu, paziņot par to
                Console.WriteLine("Nav bērnu.");
                return;
            }

            // Izvadīt katra bērna nosaukumu atkarībā no tā tipa
            foreach (var berns in berni) {
                if (berns.irDatne()) {
                    Console.WriteLine($"~{berns.nosaukums}~");
                }
                else {
                    Console.WriteLine($"/{berns.nosaukums}");
                }
            }
        }

        // Dzēš mapi vai datni
        public void dzest(string nosaukums, bool realsFails = false) {
            // Atrod dzēšamo objektu bērnu sarakstā
            var dzesamais = berni.FirstOrDefault(b => b.nosaukums.Equals(nosaukums, StringComparison.OrdinalIgnoreCase));

            if (dzesamais == null) {
                Console.WriteLine($"Kļūda: '{nosaukums}' netika atrasts.");
                return;
            }

            // Ja dzēšamais objekts ir mape
            if (dzesamais.irMape()) {
                // Ja mapei ir bērni, jautāt apstiprinājumu dzēšanai
                if (dzesamais.berni.Any()) {
                    Console.Write(
                        $"Mape '{dzesamais.nosaukums}' satur bērnus. Vai tiešām vēlaties to dzēst? (Y/n): ");

                    var atbilde = Console.ReadLine();
                    if (!atbilde.Equals("Y", StringComparison.OrdinalIgnoreCase)) {
                        Console.WriteLine("Dzēšana atcelta.");
                        return;
                    }
                }

                // Dzēš mapi no bērnu saraksta
                berni.Remove(dzesamais);
                Console.WriteLine($"Mape '{dzesamais.nosaukums}' ir izdzēsta.");
            }
            else if (dzesamais.irDatne()) { // Ja dzēšamais objekts ir datne
                // Ja ir norādīts, ka tā ir reālas datnes dzēšana, mēģināt dzēst failu no diska
                if (realsFails) {
                    var filePath = Path.Combine(Environment.CurrentDirectory, dzesamais.nosaukums);
                    try {
                        if (File.Exists(filePath)) {
                            File.Delete(filePath);
                            Console.WriteLine($"Reālais fails '{filePath}' izdzēsts no diska.");
                        }
                    }
                    catch (Exception ex) {
                        Console.WriteLine($"Kļūda dzēšot reālo failu '{filePath}': {ex.Message}");
                    }
                }

                // Dzēš datni no bērnu saraksta
                berni.Remove(dzesamais);
                Console.WriteLine($"Datne '{dzesamais.nosaukums}' ir izdzēsta.");
            }
        }

        // Izveido jaunu bērnu (mapi vai datni), ja fails ir mape
        public void izveidot(bool mape, string nosaukums) {
            // Pārbauda, vai šis fails ir mape, lai varētu pievienot bērnus
            if (irDatne()) {
                Console.WriteLine("Kļūda: Nevar izveidot bērnus datnē.");
                return;
            }

            // Pārbauda, vai nosaukums jau eksistē
            if (berni.Any(b => b.nosaukums.Equals(nosaukums, StringComparison.OrdinalIgnoreCase))) {
                Console.WriteLine($"Kļūda: Fails vai mape ar nosaukumu '{nosaukums}' jau eksistē.");
                return;
            }

            // Izveido jaunu bērnu objektu
            Fails berns = new Fails(nosaukums, mape ? FailaTips.MAPE : FailaTips.DATNE);
            berns.vecaks = this; // Uzstāda jaunā bērna vecāku
            berni.Add(berns); // Pievieno bērnu sarakstam
            Console.WriteLine($"{(mape ? "Mape" : "Datne")} '{nosaukums}' izveidota.");
        }
    }

    // Galvenā programma
    class Program {
        // Šobrīdējā datne vai mape (sākuma lokācija ir "c:")
        private static Fails sobridejais = new("c:", FailaTips.MAPE);

        // Galvenā programmas izpildes metode
        public static void Main() {
            // cikls, kas apstrādā visu
            while (true) {
                // Izvada pašreizējo atrašanās vietu un komandas uzvedni
                Console.Write(lokacija(sobridejais) + "> ");
                var ievade = Console.ReadLine(); // Nolasa lietotāja ievadi
                var ievadeMas = ievade.Split(' ', StringSplitOptions.RemoveEmptyEntries); // Sadalīt ievadi pa atstarpēm

                // Pārbauda, vai ievade nav tukša
                if (ievadeMas.Length == 0) {
                    continue;
                }

                var komanda = ievadeMas[0].ToLower(); // Pirmā daļa ir komanda

                try {
                    switch (komanda) {
                        case "by":
                        case "exit":
                            Environment.Exit(0); // Beidz programmas izpildi
                            break;
                        case "dir":
                            // Pārbauda, vai nav papildu argumenti (dir > datne)
                            if (ievadeMas.Length > 2 && ievadeMas[1].ToLower() == ">") {
                                var fileName = ievadeMas[2];
                                // Izveido reālu datni ar bērnu sarakstu
                                izveidot(sobridejais, fileName);
                            }
                            else if (ievadeMas.Length == 1) {
                                sobridejais.paradit(); // Parāda pašreizējās mapes bērnus
                            }
                            else {
                                Console.WriteLine(
                                    "Nederīga 'dir' komanda. Lietošana: dir vai dir > [datnes_nosaukums]");
                            }

                            break;

                        case "mkdir":
                            if (ievadeMas.Length < 2) { // Pārbauda, vai ir norādīts mapes nosaukums
                                Console.WriteLine("Lietošana: mkdir [mapes_nosaukums1] [mapes_nosaukums2] ...");
                            }
                            else {
                                // Iet cauri visiem norādītajiem mapju nosaukumiem un izveido tos
                                for (var i = 1; i < ievadeMas.Length; i++) {
                                    sobridejais.izveidot(true, ievadeMas[i]);
                                }
                            }

                            break;

                        case "create":
                            if (ievadeMas.Length < 2) { // Pārbauda, vai ir norādīts datnes nosaukums
                                Console.WriteLine("Lietošana: create [datnes_nosaukums1] [datnes_nosaukums2] ...");
                            }
                            else {
                                // Iet cauri visiem norādītajiem datņu nosaukumiem un izveido tos
                                for (var i = 1; i < ievadeMas.Length; i++) {
                                    sobridejais.izveidot(false, ievadeMas[i]);
                                }
                            }

                            break;

                        case "rm":
                            // Pārbauda, vai ir norādīts mapes nosaukums
                            if (ievadeMas.Length < 2) {
                                Console.WriteLine("Lietošana: rm [mapes_nosaukums1] [mapes_nosaukums2] ...");
                            }
                            else {
                                // Iet cauri visiem norādītajiem mapju nosaukumiem un dzēš tos
                                for (var i = 1; i < ievadeMas.Length; i++) {
                                    sobridejais.dzest(ievadeMas[i]);
                                }
                            }

                            break;

                        case "del":
                            // Pārbauda, vai ir norādīts datnes nosaukums
                            if (ievadeMas.Length < 2) {
                                Console.WriteLine("Lietošana: del [datnes_nosaukums1] [datnes_nosaukums2] ...");
                            }
                            else {
                                // Iet cauri visiem norādītajiem datņu nosaukumiem un dzēš tos
                                for (var i = 1; i < ievadeMas.Length; i++) {
                                    sobridejais.dzest(ievadeMas[i], true); // Dzēš gan virtuālo, gan reālo
                                }
                            }

                            break;

                        case "cd":
                            if (ievadeMas.Length < 2) { // Pārbauda, vai ir norādīts mērķis pārvietošanai
                                Console.WriteLine("Lietošana: cd [mape] vai cd .. vai cd /");
                            }
                            else {
                                // Pārvietojas uz norādīto mapi
                                parvietoties(ievadeMas[1]);
                            }

                            break;
                        case "edit":
                            if (ievadeMas.Length < 2) { // Pārbauda, vai ir norādīts datnes nosaukums
                                Console.WriteLine("Lietošana: edit [datnes_nosaukums]");
                            }
                            else {
                                // Rediģē datnes saturu
                                rediget(ievadeMas[1]);
                            }
                            break;

                        default:
                            Console.WriteLine($"Nezināma komanda: '{komanda}'.");
                            break;
                    }
                }
                catch (IndexOutOfRangeException) {
                    Console.WriteLine("Kļūda: Nepareizs komandas arguments. Lūdzu, pārbaudiet sintaksi.");
                }
                catch (Exception ex) {
                    Console.WriteLine($"Neparedzēta kļūda: {ex.Message}");
                }
            }
        }

        // Pārvietojas pa mapēm
        private static void parvietoties(string merkis) {
            // Pārvietojas uz vecāku mapi
            if (merkis == "..") {
                if (sobridejais.vecaks != null) {
                    sobridejais = sobridejais.vecaks;
                }
                else {
                    Console.WriteLine("Jūs jau atrodaties saknes mapē.");
                }
            }
            // Pārvietojas uz saknes mapi "c:"
            else if (merkis == "/") {
                while (sobridejais.vecaks != null) {
                    sobridejais = sobridejais.vecaks;
                }
            }
            // Pārvietojas uz norādīto mapi
            else {
                var targetMape = sobridejais.berni.FirstOrDefault(b => b.nosaukums.Equals(merkis, StringComparison.OrdinalIgnoreCase) && b.irMape());
                if (targetMape != null) {
                    sobridejais = targetMape;
                }
                else {
                    Console.WriteLine($"Kļūda: Mape '{merkis}' netika atrasta pašreizējā direktorijā vai nav mape.");
                }
            }
        }

        // Atgriež pilnu pašreizējās mapes ceļu
        private static string lokacija(Fails fails) {
            var pathParts = new Stack<string>();
            var temp = fails;

            while (temp != null) {
                pathParts.Push(temp.nosaukums);
                temp = temp.vecaks;
            }

            // Pārbauda, vai ceļš sākas ar "c:", lai nerādītu "c:/c:"
            if (pathParts.Peek().Equals("c:", StringComparison.OrdinalIgnoreCase) && pathParts.Count > 1) {
                pathParts.Pop(); // Izņem "c:" no steka, ja tas ir pirmais elements un nav vienīgais
            }

            return string.Join("/", pathParts).Replace("c:/", "c:\\"); // Nomaina slīpsvītras uz atpakaļvērstām slīpsvītrām, lai atbilstu Windows ceļam
        }

        // Izveido reālu datni ar 'dir' komandas izvadu
        private static void izveidot(Fails fails, string nosaukums) {
            // Pārbauda, vai tāds fails jau eksistē virtuālajā sistēmā
            var existingFile = fails.berni.FirstOrDefault(b =>
                b.nosaukums.Equals(nosaukums, StringComparison.OrdinalIgnoreCase) && b.irDatne());

            // Ja fails eksistē, izdzēš to pirms jauna izveidošanas
            if (existingFile != null) {
                fails.dzest(nosaukums, true); // True norāda, ka jādzēš arī reālais fails
            }

            // Izveido jaunu virtuālo datni
            fails.izveidot(false, nosaukums);

            // Veido faila ceļu reālajā sistēmā
            var filePath = Path.Combine(Environment.CurrentDirectory, nosaukums);

            try {
                using (var sw = new StreamWriter(filePath)) {
                    if (fails.berni.Count == 0) {
                        sw.WriteLine("Nav bērnu.");
                    }
                    else {
                        foreach (var berns in fails.berni) {
                            sw.WriteLine(berns.irDatne() ? $"~{berns.nosaukums}~" : $"/{berns.nosaukums}");
                        }
                    }
                }

                Console.WriteLine($"'dir' izvads saglabāts reālā datnē: {filePath}");
            }
            catch (Exception ex) {
                Console.WriteLine($"Kļūda, saglabājot 'dir' izvadu reālā datnē: {ex.Message}");
            }
        }

        // Rediģē datnes saturu (izvada datnes nosaukumu un reālās datnes saturu)
        private static void rediget(string nosaukums) {
            var fails = sobridejais.berni.FirstOrDefault(b => b.nosaukums.Equals(nosaukums, StringComparison.OrdinalIgnoreCase) && b.irDatne());

            if (fails == null) {
                Console.WriteLine($"Kļūda: Datne '{nosaukums}' netika atrasta.");
                return;
            }

            Console.WriteLine($"Datnes nosaukums: {fails.nosaukums}");
            var filePath = Path.Combine(Environment.CurrentDirectory, nosaukums);

            try {
                if (File.Exists(filePath)) {
                    // Izlasa un izvada faila saturu
                    Console.WriteLine("\n--- Datnes saturs ---");
                    Console.WriteLine(File.ReadAllText(filePath));
                    Console.WriteLine("---------------------\n");
                }
                else {
                    Console.WriteLine($"Reālā datne '{filePath}' neeksistē. Nevar parādīt saturu.");
                }
            }
            catch (Exception ex) {
                Console.WriteLine($"Kļūda lasot reālās datnes saturu: {ex.Message}");
            }
        }
    }
}