// Importējam React hookus stāvokļa un dzīves cikla pārvaldībai
import { useEffect, useMemo, useState } from 'react'

// Importējam aplikācijas CSS stilus
import './App.css'

function App() {
  // Pašreizējā lapa: 'main' (galvenā) vai 'auth' (autentifikācija)
  const [page, setPage] = useState('main')

  // Lietotāja ievades lauki
  const [username, setUsername] = useState('')
  const [password, setPassword] = useState('')

  // Lietotāja pieslēgšanās statuss
  const [loggedIn, setLoggedIn] = useState(false)

  // Sistēmas paziņojumi lietotājam
  const [message, setMessage] = useState('')

  // Informācija par pēdējo aktivitāti (no DB)
  const [lastActivity, setLastActivity] = useState('')

  // Backend API bāzes adrese
  // Tiek ņemta no Vite vides mainīgā vai izmantota noklusētā vērtība
  const apiBase = useMemo(
    () => import.meta.env.VITE_API_URL || 'http://localhost:3001',
    []
  )

  // Universāla funkcija API izsaukumiem
  // Frontend sazinās tikai ar backend maršrutiem,
  // backend savukārt izsauc PostgreSQL funkcijas
  const callApi = async (path, payload) => {
    const response = await fetch(`${apiBase}${path}`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(payload),
    })
    return response.json()
  }

  // Lietotāja statusa sinhronizācija ar serveri
  // Backend pārbauda sesijas derīgumu (2 minūšu taimauts DB līmenī)
  const syncStatus = async (storedUsername) => {
    try {
      const data = await callApi('/status', { username: storedUsername })
      if (data?.isLoggedIn) {
        setLoggedIn(true)
        setUsername(storedUsername)
      } else {
        setLoggedIn(false)
        localStorage.removeItem('username')
      }
    } catch (error) {
      setLoggedIn(false)
      setMessage('Neizdevās pārbaudīt statusu')
    }
  }

  // Komponents ielādes brīdī pārbauda,
  // vai localStorage ir saglabāts lietotājs
  useEffect(() => {
    const storedUsername = localStorage.getItem('username')
    if (storedUsername) {
      syncStatus(storedUsername)
    }
  }, [])

  // Lietotāja reģistrācija
  // Izsauc backend maršrutu /register,
  // kurš savukārt izsauc DB funkciju create_user
  const handleRegister = async () => {
    setMessage('')
    try {
      const data = await callApi('/register', { username, password })
      setMessage(data?.message || 'Nezināma atbilde')
    } catch (error) {
      setMessage('Reģistrācija neizdevās')
    }
  }

  // Lietotāja pieteikšanās
  // Backend pārbauda paroli DB līmenī un izveido sesiju
  const handleLogin = async () => {
    setMessage('')
    try {
      const data = await callApi('/login', { username, password })
      if (data?.isLoggedIn) {
        localStorage.setItem('username', data.username)
        setLoggedIn(true)
        setPage('main')
      }
      setMessage(data?.message || 'Nezināma atbilde')
    } catch (error) {
      setMessage('Pieslēgšanās neizdevās')
    }
  }

  // Lietotāja izlogošanās
  // Izsauc backend maršrutu /logout,
  // kas deaktivizē sesiju DB pusē
  const handleLogout = async () => {
    setMessage('')
    setLastActivity('')
    try {
      await callApi('/logout', { username })
      setLoggedIn(false)
      localStorage.removeItem('username')
      setMessage('Veiksmīgi izlogojies')
    } catch (error) {
      setMessage('Izrakstīšanās neizdevās')
    }
  }

  // Aizsargāta darbība
  // Izsauc DB funkciju perform_action,
  // kas atjauno sesijas laiku tikai aktīvam lietotājam
  const handleAction = async () => {
    setMessage('')
    try {
      const data = await callApi('/action', { username })

      // Ja sesija ir beigusies, lietotājs tiek automātiski izlogots
      if (!data?.allowed && data?.message === 'Sesijas laiks beidzies') {
        setLoggedIn(false)
        localStorage.removeItem('username')
      }

      // Pēdējās aktivitātes laika formatēšana
      if (data?.lastActivity) {
        const formatted = new Date(data.lastActivity).toLocaleString('lv-LV')
        setLastActivity(`Pēdējā aktivitāte: ${formatted}`)
      } else {
        setLastActivity('')
      }

      setMessage(data?.message || 'Nezināma atbilde')
    } catch (error) {
      setMessage('Darbība neizdevās')
    }
  }

  return (
    <div className="app-shell w3-container">
      {/* Autentifikācijas lapa */}
      {page === 'auth' ? (
        <div className="w3-card w3-padding w3-white auth-card">
          <h2 className="w3-center">Pieslēgšanās vai reģistrācija</h2>

          <label className="w3-text-grey">Lietotājvārds</label>
          <input
            className="w3-input w3-border w3-margin-bottom"
            value={username}
            onChange={(event) => setUsername(event.target.value)}
          />

          <label className="w3-text-grey">Parole</label>
          <input
            className="w3-input w3-border w3-margin-bottom"
            type="password"
            value={password}
            onChange={(event) => setPassword(event.target.value)}
          />

          <div className="w3-row-padding">
            <div className="w3-half">
              <button className="w3-button w3-blue w3-block" onClick={handleRegister}>
                Reģistrēt
              </button>
            </div>
            <div className="w3-half">
              <button className="w3-button w3-green w3-block" onClick={handleLogin}>
                Pieslēgties
              </button>
            </div>
          </div>

          {message ? <p className="w3-center w3-margin-top">{message}</p> : null}
        </div>
      ) : (
        /* Galvenā lapa */
        <div className="w3-card w3-padding w3-white main-card">
          <h2 className="w3-center">Galvenā lapa</h2>

          {/* Lietotāja statuss */}
          <p className="w3-center">
            {loggedIn ? `Lietotājs: ${username}` : 'Viesis'}
          </p>

          <div className="w3-row-padding w3-margin-top">
            <div className="w3-third">
              <button className="w3-button w3-blue w3-block" onClick={() => setPage('auth')}>
                Pieslēgties
              </button>
            </div>

            <div className="w3-third">
              <button
                className="w3-button w3-red w3-block"
                onClick={handleLogout}
                disabled={!loggedIn}
              >
                Iziet
              </button>
            </div>

            <div className="w3-third">
              <button className="w3-button w3-teal w3-block" onClick={handleAction}>
                Darbība
              </button>
            </div>
          </div>

          {message ? <p className="w3-center w3-margin-top">{message}</p> : null}
          {lastActivity ? <p className="w3-center">{lastActivity}</p> : null}
        </div>
      )}
    </div>
  )
}

export default App