-- =========================================================
-- DATUBĀZES SHĒMA UN BIZNESA LOĢIKA
-- =========================================================
-- Visa autentifikācijas un sesiju loģika tiek realizēta
-- TIKAI PostgreSQL funkcijās.
--
-- Backend NEKAD neveic tiešus:
-- INSERT / UPDATE / SELECT uz tabulām.
-- Backend tikai IZSAUC šīs DB funkcijas.
-- =========================================================

-- Paplašinājums paroles šifrēšanai (crypt, gen_salt)
CREATE EXTENSION IF NOT EXISTS pgcrypto;

-- =========================================================
-- LIETOTĀJU TABULA
-- Glabā lietotājvārdu un paroles hešu
-- =========================================================
CREATE TABLE IF NOT EXISTS users (
  id SERIAL PRIMARY KEY,              -- Unikāls lietotāja ID
  username TEXT UNIQUE NOT NULL,       -- Lietotājvārds (unikāls)
  password_hash TEXT NOT NULL          -- Hešota parole (nekad netiek glabāta atklātā veidā)
);

-- =========================================================
-- SESIJU TABULA
-- Glabā pieteikšanās statusu un pēdējo aktivitāti
-- =========================================================
CREATE TABLE IF NOT EXISTS sessions (
  user_id INTEGER PRIMARY KEY          -- Viens lietotājs = viena sesija
    REFERENCES users(id) ON DELETE CASCADE,
  is_logged_in BOOLEAN NOT NULL DEFAULT FALSE, -- Pieteikšanās statuss
  last_activity TIMESTAMPTZ             -- Pēdējās aktivitātes laiks
);

-- =========================================================
-- 1) LIETOTĀJA IZVEIDE
-- Parole tiek hešota DB pusē
-- =========================================================
CREATE OR REPLACE FUNCTION create_user(p_username TEXT, p_password TEXT)
RETURNS TEXT
LANGUAGE plpgsql
AS $$
DECLARE
  v_user_id INTEGER; -- Saglabā jaunā lietotāja ID
BEGIN
  -- Lietotājvārda validācija
  IF p_username IS NULL OR length(trim(p_username)) = 0 THEN
    RETURN 'Lietotājvārds nav derīgs';
  END IF;

  -- Paroles validācija
  IF p_password IS NULL OR length(p_password) = 0 THEN
    RETURN 'Parole nav derīga';
  END IF;

  -- Pārbaude, vai lietotājs jau eksistē
  IF EXISTS (SELECT 1 FROM users WHERE username = p_username) THEN
    RETURN 'Lietotājs jau eksistē';
  END IF;

  -- Jauna lietotāja ievietošana ar paroles hešu
  INSERT INTO users (username, password_hash)
  VALUES (p_username, crypt(p_password, gen_salt('bf')))
  RETURNING id INTO v_user_id;

  -- Izveidojam sākotnējo sesiju (nepieslēgts)
  INSERT INTO sessions (user_id, is_logged_in, last_activity)
  VALUES (v_user_id, FALSE, NULL);

  RETURN 'Lietotājs izveidots';

-- Kļūdu apstrāde DB līmenī
EXCEPTION
  WHEN OTHERS THEN
    RETURN 'Kļūda: ' || SQLERRM;
END;
$$;

-- =========================================================
-- 2) LIETOTĀJA DZĒŠANA
-- =========================================================
CREATE OR REPLACE FUNCTION delete_user(p_username TEXT)
RETURNS TEXT
LANGUAGE plpgsql
AS $$
BEGIN
  -- Dzēš lietotāju (sesija dzēsīsies automātiski ar CASCADE)
  DELETE FROM users WHERE username = p_username;

  -- Ja lietotājs nav atrasts
  IF NOT FOUND THEN
    RETURN 'Lietotājs nav atrasts';
  END IF;

  RETURN 'Lietotājs dzēsts';
EXCEPTION
  WHEN OTHERS THEN
    RETURN 'Kļūda: ' || SQLERRM;
END;
$$;

-- =========================================================
-- 3) PAROLES ATJAUNINĀŠANA
-- =========================================================
CREATE OR REPLACE FUNCTION update_user(p_username TEXT, p_new_password TEXT)
RETURNS TEXT
LANGUAGE plpgsql
AS $$
BEGIN
  -- Jaunās paroles validācija
  IF p_new_password IS NULL OR length(p_new_password) = 0 THEN
    RETURN 'Parole nav derīga';
  END IF;

  -- Atjaunojam paroles hešu
  UPDATE users
  SET password_hash = crypt(p_new_password, gen_salt('bf'))
  WHERE username = p_username;

  IF NOT FOUND THEN
    RETURN 'Lietotājs nav atrasts';
  END IF;

  -- Pēc paroles maiņas lietotājs tiek izlogots
  UPDATE sessions
  SET is_logged_in = FALSE,
      last_activity = NULL
  WHERE user_id = (SELECT id FROM users WHERE username = p_username);

  RETURN 'Parole atjaunināta';
EXCEPTION
  WHEN OTHERS THEN
    RETURN 'Kļūda: ' || SQLERRM;
END;
$$;

-- =========================================================
-- 4) PIETEIKŠANĀS (LOGIN)
-- =========================================================
CREATE OR REPLACE FUNCTION login_user(p_username TEXT, p_password TEXT)
RETURNS TABLE(username TEXT, is_logged_in BOOLEAN, message TEXT)
LANGUAGE plpgsql
AS $$
DECLARE
  v_user_id INTEGER; -- Lietotāja ID
  v_hash TEXT;       -- Saglabātais paroles hešs
BEGIN
  -- Iegūstam lietotāja datus
  SELECT id, password_hash
  INTO v_user_id, v_hash
  FROM users
  WHERE users.username = p_username;

  -- Ja lietotājs neeksistē
  IF v_user_id IS NULL THEN
    RETURN QUERY SELECT p_username, FALSE, 'Lietotājs nav atrasts';
    RETURN;
  END IF;

  -- Paroles pārbaude
  IF v_hash IS NULL OR crypt(p_password, v_hash) <> v_hash THEN
    RETURN QUERY SELECT p_username, FALSE, 'Nepareiza parole';
    RETURN;
  END IF;

  -- Atjaunojam sesijas statusu
  UPDATE sessions
  SET is_logged_in = TRUE,
      last_activity = NOW()
  WHERE user_id = v_user_id;

  -- Ja sesija vēl neeksistē
  IF NOT FOUND THEN
    INSERT INTO sessions (user_id, is_logged_in, last_activity)
    VALUES (v_user_id, TRUE, NOW());
  END IF;

  RETURN QUERY SELECT p_username, TRUE, 'Veiksmīgi pieslēdzies';
EXCEPTION
  WHEN OTHERS THEN
    RETURN QUERY SELECT p_username, FALSE, 'Kļūda: ' || SQLERRM;
END;
$$;

-- =========================================================
-- 5) IZLOGOŠANĀS
-- =========================================================
CREATE OR REPLACE FUNCTION logout_user(p_username TEXT)
RETURNS TEXT
LANGUAGE plpgsql
AS $$
DECLARE
  v_user_id INTEGER;
BEGIN
  -- Atrodam lietotāja ID
  SELECT id INTO v_user_id FROM users WHERE username = p_username;

  IF v_user_id IS NULL THEN
    RETURN 'Lietotājs nav atrasts';
  END IF;

  -- Atjaunojam sesijas statusu
  UPDATE sessions
  SET is_logged_in = FALSE,
      last_activity = NOW()
  WHERE user_id = v_user_id;

  RETURN 'Veiksmīgi izlogojies';
EXCEPTION
  WHEN OTHERS THEN
    RETURN 'Kļūda: ' || SQLERRM;
END;
$$;

-- =========================================================
-- 6) SESIJAS STATUSA PĀRBAUDE AR TIMEOUT
-- =========================================================
CREATE OR REPLACE FUNCTION is_user_logged_in(p_username TEXT)
RETURNS BOOLEAN
LANGUAGE plpgsql
AS $$
DECLARE
  v_user_id INTEGER;
  v_is_logged_in BOOLEAN;
  v_last_activity TIMESTAMPTZ;
  v_timeout INTERVAL := INTERVAL '2 minutes'; -- Sesijas noilgums
BEGIN
  -- Iegūstam lietotāja ID
  SELECT id INTO v_user_id FROM users WHERE username = p_username;

  IF v_user_id IS NULL THEN
    RETURN FALSE;
  END IF;

  -- Nolasa sesijas informāciju
  SELECT is_logged_in, sessions.last_activity
  INTO v_is_logged_in, v_last_activity
  FROM sessions
  WHERE user_id = v_user_id;

  -- Ja lietotājs nav pieslēdzies
  IF v_is_logged_in IS DISTINCT FROM TRUE THEN
    RETURN FALSE;
  END IF;

  -- Ja sesija ir beigusies
  IF v_last_activity IS NULL OR NOW() - v_last_activity > v_timeout THEN
    UPDATE sessions
    SET is_logged_in = FALSE
    WHERE user_id = v_user_id;
    RETURN FALSE;
  END IF;

  RETURN TRUE;
EXCEPTION
  WHEN OTHERS THEN
    RETURN FALSE;
END;
$$;

-- =========================================================
-- 7) AIZSARGĀTA DARBĪBA
-- Atjauno last_activity tikai aktīvai sesijai
-- =========================================================
CREATE OR REPLACE FUNCTION perform_action(p_username TEXT)
RETURNS TABLE(allowed BOOLEAN, message TEXT, last_activity TIMESTAMPTZ)
LANGUAGE plpgsql
AS $$
DECLARE
  v_user_id INTEGER;
  v_is_logged_in BOOLEAN;
  v_last_activity TIMESTAMPTZ;
  v_timeout INTERVAL := INTERVAL '2 minutes';
BEGIN
  -- Iegūstam lietotāja ID
  SELECT id INTO v_user_id FROM users WHERE username = p_username;

  IF v_user_id IS NULL THEN
    RETURN QUERY SELECT FALSE, 'Lietotājs nav atrasts', NULL::timestamptz;
    RETURN;
  END IF;

  -- Nolasa sesijas stāvokli
  SELECT is_logged_in, sessions.last_activity
  INTO v_is_logged_in, v_last_activity
  FROM sessions
  WHERE user_id = v_user_id;

  -- Ja nav pieslēdzies
  IF v_is_logged_in IS DISTINCT FROM TRUE THEN
    RETURN QUERY SELECT FALSE, 'Nav pieslēdzies', NULL::timestamptz;
    RETURN;
  END IF;

  -- Ja sesijas laiks ir beidzies
  IF v_last_activity IS NULL OR NOW() - v_last_activity > v_timeout THEN
    UPDATE sessions
    SET is_logged_in = FALSE
    WHERE user_id = v_user_id;
    RETURN QUERY SELECT FALSE, 'Sesijas laiks beidzies', NULL::timestamptz;
    RETURN;
  END IF;

  -- Atjaunojam pēdējās aktivitātes laiku
  UPDATE sessions
  SET last_activity = NOW()
  WHERE user_id = v_user_id;

  RETURN QUERY SELECT TRUE, 'Darbība veikta', NOW();
EXCEPTION
  WHEN OTHERS THEN
    RETURN QUERY SELECT FALSE, 'Kļūda: ' || SQLERRM, NULL::timestamptz;
END;
$$;