// Importējam Express ietvaru HTTP servera izveidei
const express = require('express');

// Importējam CORS, lai frontend varētu sazināties ar backend
const cors = require('cors');

// Importējam dotenv, lai nolasītu .env failu
const dotenv = require('dotenv');

// Ielādējam vides mainīgos no .env faila
dotenv.config();

// Importējam PostgreSQL savienojumu (Pool)
// ŠIS POOL TIEK LIETOTS TIKAI DB FUNKCIJU IZSAUKŠANAI
// NAV tiešas piekļuves tabulām
const pool = require('./db');

// Inicializējam Express aplikāciju
const app = express();

// Servera ports (ņemts no .env vai noklusēti 3001)
const port = process.env.PORT || 3001;

// Atļaujam pieprasījumus no citām adresēm (frontend)
app.use(cors());

// Ļaujam lasīt JSON datus no request body
app.use(express.json());

// Palīgfunkcija ievades datu pārbaudei
// Pārbauda, vai request body satur visus nepieciešamos laukus
const requireFields = (fields, body) =>
  fields.every((field) => body?.[field]);

/* ============================
   LIETOTĀJA REĢISTRĀCIJA
   ============================ */
app.post('/register', async (req, res) => {
  // Pārbaudām, vai ir ievadīts lietotājvārds un parole
  if (!requireFields(['username', 'password'], req.body)) {
    return res.status(400).json({
      ok: false,
      message: 'Trūkst lietotāja datu',
    });
  }

  try {
    // Izsaucam PostgreSQL funkciju create_user
    // ⚠️ NAV tiešas INSERT darbības uz tabulām
    const result = await pool.query(
      'SELECT create_user($1, $2) AS message',
      [req.body.username, req.body.password]
    );

    // Atgriežam DB funkcijas atbildi frontend
    return res.json({
      ok: true,
      message: result.rows[0]?.message,
    });
  } catch (error) {
    // Servera kļūdu apstrāde
    return res.status(500).json({
      ok: false,
      message: 'Servera kļūda',
      error: error.message,
    });
  }
});

/* ============================
   LIETOTĀJA PIETEIKŠANĀS
   ============================ */
app.post('/login', async (req, res) => {
  // Pārbaudām ievades laukus
  if (!requireFields(['username', 'password'], req.body)) {
    return res.status(400).json({
      ok: false,
      message: 'Trūkst lietotāja datu',
    });
  }

  try {
    // Izsaucam PostgreSQL funkciju login_user
    // DB funkcija pārbauda paroli un sesijas statusu
    const result = await pool.query(
      'SELECT * FROM login_user($1, $2)',
      [req.body.username, req.body.password]
    );

    const row = result.rows[0];

    // Atgriežam pieteikšanās statusu
    return res.json({
      ok: true,
      username: row?.username ?? req.body.username,
      isLoggedIn: row?.is_logged_in ?? false,
      message: row?.message ?? 'Nezināma atbilde',
    });
  } catch (error) {
    return res.status(500).json({
      ok: false,
      message: 'Servera kļūda',
      error: error.message,
    });
  }
});

/* ============================
   LIETOTĀJA IZLOGOŠANĀS
   ============================ */
app.post('/logout', async (req, res) => {
  // Pārbaudām, vai ir ievadīts lietotājvārds
  if (!requireFields(['username'], req.body)) {
    return res.status(400).json({
      ok: false,
      message: 'Trūkst lietotājvārda',
    });
  }

  try {
    // Izsaucam DB funkciju logout_user
    const result = await pool.query(
      'SELECT logout_user($1) AS message',
      [req.body.username]
    );

    return res.json({
      ok: true,
      message: result.rows[0]?.message,
    });
  } catch (error) {
    return res.status(500).json({
      ok: false,
      message: 'Servera kļūda',
      error: error.message,
    });
  }
});

/* ============================
   LIETOTĀJA STATUSA PĀRBAUDE
   ============================ */
app.post('/status', async (req, res) => {
  if (!requireFields(['username'], req.body)) {
    return res.status(400).json({
      ok: false,
      message: 'Trūkst lietotājvārda',
    });
  }

  try {
    // Sesijas derīgumu (timeout) apstrādā DB funkcija
    // Backend tikai nolasa rezultātu
    const result = await pool.query(
      'SELECT is_user_logged_in($1) AS is_logged_in',
      [req.body.username]
    );

    return res.json({
      ok: true,
      isLoggedIn: result.rows[0]?.is_logged_in ?? false,
      username: req.body.username,
    });
  } catch (error) {
    return res.status(500).json({
      ok: false,
      message: 'Servera kļūda',
      error: error.message,
    });
  }
});

/* ============================
   DARBĪBAS SIMULĀCIJA
   ============================ */
app.post('/action', async (req, res) => {
  if (!requireFields(['username'], req.body)) {
    return res.status(400).json({
      ok: false,
      message: 'Trūkst lietotājvārda',
    });
  }

  try {
    // Izsaucam DB funkciju perform_action
    // Tā atjauno sesijas laiku, ja lietotājs ir pieslēdzies
    const result = await pool.query(
      'SELECT * FROM perform_action($1)',
      [req.body.username]
    );

    const row = result.rows[0];

    return res.json({
      ok: true,
      allowed: row?.allowed ?? false,
      message: row?.message ?? 'Nezināma atbilde',
      lastActivity: row?.last_activity ?? null,
    });
  } catch (error) {
    return res.status(500).json({
      ok: false,
      message: 'Servera kļūda',
      error: error.message,
    });
  }
});

// Palaižam serveri
// Backend NESTRĀDĀ ar tabulām tieši,
// tikai izsauc PostgreSQL funkcijas
app.listen(port, () => {
  console.log(`Serveris darbojas uz porta ${port}`);
});