<?php
session_start();
require_once '../includes/config.php';
require_once '../includes/database.php';
require_once '../includes/auth.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || !$auth->hasRole('treneris')) {
    header("Location: ../index.php");
    exit();
}

$lietotajs = $auth->getCurrentUser();
$db = Database::getInstance();

// Dzēšanas funkcija (tikai savu inventāru - nevis sistēmas)
if (isset($_GET['dzest'])) {
    $id = intval($_GET['dzest']);
    
    // Pārbaudīt vai inventārs pieder šim trenerim (nav sistēmas)
    $check = $db->izpilditVaicajumu("SELECT * FROM inventars WHERE id = ? AND pievienoja_id = ?", [$id, $lietotajs['id']]);
    if ($check->fetch()) {
        $nosaukums = $check->fetch()['nosaukums'];
        $db->izpilditVaicajumu("DELETE FROM inventars WHERE id = ?", [$id]);
        
        // Žurnāls
        $db->izpilditVaicajumu(
            "INSERT INTO zurnals (lietotaja_id, darbiba, ip_adrese) VALUES (?, ?, ?)",
            [$lietotajs['id'], 'Izdzēsa inventāru: ' . $nosaukums, $_SERVER['REMOTE_ADDR'] ?? '']
        );
        
        header("Location: inventars.php?success=3");
        exit();
    } else {
        header("Location: inventars.php?error=1");
        exit();
    }
}

// Pievienošanas funkcija
if (isset($_POST['pievienot'])) {
    $nosaukums = trim($_POST['nosaukums']);
    $apraksts = trim($_POST['apraksts']);
    $daudzums = intval($_POST['daudzums']);
    $stavoklis = $_POST['stavoklis'];
    
    // Validācija
    if (empty($nosaukums) || $daudzums < 0) {
        $error = "Lūdzu aizpildiet visus obligātos laukus!";
    } else {
        try {
            // Mēģinām pievienot ar pievienoja_id
            try {
                $db->izpilditVaicajumu(
                    "INSERT INTO inventars (nosaukums, apraksts, daudzums, stavoklis, pirkuma_datums, pievienoja_id) VALUES (?, ?, ?, ?, NOW(), ?)",
                    [$nosaukums, $apraksts, $daudzums, $stavoklis, $lietotajs['id']]
                );
            } catch (Exception $e) {
                // Ja kolonna neeksistē, pievienojam bez tās
                $db->izpilditVaicajumu(
                    "INSERT INTO inventars (nosaukums, apraksts, daudzums, stavoklis, pirkuma_datums) VALUES (?, ?, ?, ?, NOW())",
                    [$nosaukums, $apraksts, $daudzums, $stavoklis]
                );
            }
            
            // Žurnāls
            $db->izpilditVaicajumu(
                "INSERT INTO zurnals (lietotaja_id, darbiba, ip_adrese) VALUES (?, ?, ?)",
                [$lietotajs['id'], 'Pievienoja inventāru: ' . $nosaukums, $_SERVER['REMOTE_ADDR'] ?? '']
            );
            
            $success = "Inventārs veiksmīgi pievienots!";
            
        } catch (Exception $e) {
            $error = "Kļūda pievienojot inventāru!";
        }
    }
}

// Inventāra saraksts (ar lietotāja vārdu, ja kolonna eksistē)
try {
    $inventars = $db->izpilditVaicajumu("SELECT i.*, l.vards, l.uzvards FROM inventars i LEFT JOIN lietotaji l ON i.pievienoja_id = l.id ORDER BY i.izveidots DESC")->fetchAll();
} catch (Exception $e) {
    // Ja kolonna neeksistē, izvēlamies bez pievienotāja informācijas
    $inventars = $db->izpilditVaicajumu("SELECT *, NULL as vards, NULL as uzvards FROM inventars ORDER BY izveidots DESC")->fetchAll();
}

// Inventāra statistika
$kopais_inventars = count($inventars);
$labā_stāvoklī = count(array_filter($inventars, fn($i) => $i['stavoklis'] == 'labā stāvoklī'));
$jāremontē = count(array_filter($inventars, fn($i) => $i['stavoklis'] == 'jāremontē'));

// Veiksmes ziņojumi
$success_message = '';
if (isset($_GET['success'])) {
    switch ($_GET['success']) {
        case '3': $success_message = 'Inventārs veiksmīgi dzēsts!'; break;
    }
}
?>
<!DOCTYPE html>
<html lang="lv">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Inventārs - Trenera Panelis</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="../assets/css/style.css">
</head>
<body>
    <?php include '../includes/header.php'; ?>

    <div class="container mt-4">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1>Inventārs</h1>
        </div>

        <!-- Ziņojumi -->
        <?php if (!empty($success_message)): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $success_message; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (isset($error)): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $error; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (isset($_GET['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                Kļūda dzēšot inventāru!
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <!-- Statistika -->
        <div class="row mt-4">
            <div class="col-md-4 mb-3">
                <div class="card text-white bg-primary">
                    <div class="card-body">
                        <h5 class="card-title">Kopējais inventārs</h5>
                        <p class="card-text display-6"><?php echo $kopais_inventars; ?></p>
                    </div>
                </div>
            </div>
            <div class="col-md-4 mb-3">
                <div class="card text-white bg-success">
                    <div class="card-body">
                        <h5 class="card-title">Labā stāvoklī</h5>
                        <p class="card-text display-6"><?php echo $labā_stāvoklī; ?></p>
                    </div>
                </div>
            </div>
            <div class="col-md-4 mb-3">
                <div class="card text-white bg-warning">
                    <div class="card-body">
                        <h5 class="card-title">Jāremontē</h5>
                        <p class="card-text display-6"><?php echo $jāremontē; ?></p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Pievienošanas forma -->
        <div class="card mt-4">
            <div class="card-header bg-success text-white">
                <h5 class="mb-0">Pievienot jaunu inventāru</h5>
            </div>
            <div class="card-body">
                <form method="POST">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="nosaukums" class="form-label">Nosaukums *</label>
                            <input type="text" class="form-control" id="nosaukums" name="nosaukums" required>
                        </div>
                        <div class="col-md-3 mb-3">
                            <label for="daudzums" class="form-label">Daudzums *</label>
                            <input type="number" class="form-control" id="daudzums" name="daudzums" value="1" min="1" required>
                        </div>
                        <div class="col-md-3 mb-3">
                            <label for="stavoklis" class="form-label">Stāvoklis *</label>
                            <select class="form-select" id="stavoklis" name="stavoklis" required>
                                <option value="labā stāvoklī">Labā stāvoklī</option>
                                <option value="jāremontē">Jāremontē</option>
                                <option value="nolietots">Nolietots</option>
                            </select>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label for="apraksts" class="form-label">Apraksts</label>
                        <textarea class="form-control" id="apraksts" name="apraksts" rows="2" placeholder="Inventāra apraksts..."></textarea>
                    </div>
                    <button type="submit" name="pievienot" class="btn btn-success">Pievienot inventāru</button>
                </form>
            </div>
        </div>

        <!-- Inventāra saraksts -->
        <div class="card mt-4">
            <div class="card-header">
                <h5 class="mb-0">Inventāra saraksts</h5>
            </div>
            <div class="card-body">
                <?php if (empty($inventars)): ?>
                    <div class="alert alert-info">
                        Nav pievienots neviens inventāra priekšmets.
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Nosaukums</th>
                                    <th>Apraksts</th>
                                    <th>Daudzums</th>
                                    <th>Stāvoklis</th>
                                    <th>Pievienots</th>
                                    <th>Pievienoja</th>
                                    <th>Darbības</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($inventars as $items): ?>
                                    <?php 
                                    $is_own = $items['pievienoja_id'] == $lietotajs['id'];
                                    $is_system = empty($items['pievienoja_id']);
                                    $pievienotajs = !empty($items['vards']) ? $items['vards'] . ' ' . $items['uzvards'] : 'Sistēma';
                                    ?>
                                    <tr>
                                        <td><?php echo $items['id']; ?></td>
                                        <td><?php echo htmlspecialchars($items['nosaukums']); ?></td>
                                        <td><?php echo htmlspecialchars($items['apraksts']); ?></td>
                                        <td>
                                            <span class="badge bg-primary"><?php echo $items['daudzums']; ?></span>
                                        </td>
                                        <td>
                                            <span class="badge 
                                                <?php echo $items['stavoklis'] == 'labā stāvoklī' ? 'bg-success' : 
                                                       ($items['stavoklis'] == 'jāremontē' ? 'bg-warning' : 'bg-secondary'); ?>">
                                                <?php echo $items['stavoklis']; ?>
                                            </span>
                                        </td>
                                        <td><small><?php echo date('d.m.Y H:i', strtotime($items['izveidots'])); ?></small></td>
                                        <td><small><?php echo $pievienotajs; ?></small></td>
                                        <td>
                                            <?php if ($is_own): ?>
                                                <div class="btn-group btn-group-sm">
                                                    <a href="rediget_inventaru.php?id=<?php echo $items['id']; ?>" class="btn btn-outline-primary">Rediģēt</a>
                                                    <a href="inventars.php?dzest=<?php echo $items['id']; ?>" 
                                                       class="btn btn-outline-danger"
                                                       onclick="return confirm('Vai tiešām vēlaties dzēst šo inventāru?')">
                                                        Dzēst
                                                    </a>
                                                </div>
                                            <?php elseif ($is_system): ?>
                                                <span class="text-muted small">Sistēmas inventārs</span>
                                            <?php else: ?>
                                                <span class="text-muted small">Cits treneris</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
