<?php

namespace App\Http\Controllers;

use App\Models\Gramata;
use App\Models\UserFavorite;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class GramataController extends Controller
{
    /**
     * Izvada grāmatu sarakstu tikai autentificētiem lietotājiem
     */
    public function index()
    {
        $userId = Auth::id();
        
        // Iegūt visas grāmatas un kārto tās pēc user_id un book_id mīļākās ( favorite) grāmatas
        $gramatas = Gramata::all()->map(function ($gramata) use ($userId) {
            $gramata->is_favorite = UserFavorite::where('user_id', $userId)
                                                 ->where('book_id', $gramata->id)
                                                 ->exists();
            return $gramata;
        });

        return view('gramatas.index', compact('gramatas'));
    }

    /**
     * Filtrē grāmatas pēc favorītu statusa:
     */
    public function showFavorites(Request $request)
    {
        $userId = Auth::id();
        $showFavorites = $request->input('show_favorites', false);

        if ($showFavorites) {
            // Iegūt tikai favorītu grāmatas
            $gramatas = Gramata::all()->map(function ($gramata) use ($userId) {
                $gramata->is_favorite = UserFavorite::where('user_id', $userId)
                                                     ->where('book_id', $gramata->id)
                                                     ->exists();
                return $gramata;
            })->filter(function ($gramata) {
                return $gramata->is_favorite; // Parādīt tikai favorītus
            });

            // Atgriež JSON atbildi ar favorītu sarakstu
            return response()->json([
                'success' => true,
                'books' => view('gramatas.books', compact('gramatas'))->render(),
            ]);
        }

        // Kad show only favorites nav izvēlēts, lietotāju pārsūta uz sākuma lapu.
        return redirect()->route('gramatas.index');
    }

    /**
     * Jaunas grāmatas pievienošanas forma:
     */
    public function create()
    {
        return view('gramatas.create');
    }

    /**
     * Saglabā jaunu grāmatu datubāzē:
     */
    public function store(Request $request)
    {
        $request->validate([
            'nosaukums' => 'required|string|max:255',
            'autors' => 'required|string|max:255',
            'audio' => 'nullable|file|mimes:mp3|max:20480', // Maksimālais izmērs: 20MB
        ]);

        // Saglabāt audio failu, ja tas ir augšupielādēts ( var arī nebūt)
        $audioPath = null; // audio fails var nebūt, lapa neizvada kļūdu, ja lietotājs nav pievienojis mp3 failu
        if ($request->hasFile('audio')) {
            $audioPath = $request->file('audio')->store('audio', 'public');
        }

        // Izveidot jaunu grāmatas ierakstu
        Gramata::create([
            'nosaukums' => $request->nosaukums,
            'autors' => $request->autors,
            'audio_path' => $audioPath,
        ]);

        return redirect()->route('gramatas.index')->with('success', 'Grāmata pievienota!');
    }

    /**
     * Rādīt grāmatas rediģēšanas formu.
     */
    public function edit(Gramata $gramata)
    {
        return view('gramatas.edit', compact('gramata'));
    }

    /**
     * Atjaunināt esošu grāmatas ierakstu:
     */
    public function update(Request $request, Gramata $gramata)
    {
        $validatedData = $request->validate([
            'nosaukums' => 'required|string|max:255',
            'autors' => 'required|string|max:255',
            'audio' => 'nullable|mimes:mp3|max:20480', // Maksimālais izmērs: 20MB
        ]);

        // Atjaunināt grāmatas detaļas
        $gramata->nosaukums = $validatedData['nosaukums'];
        $gramata->autors = $validatedData['autors'];

        // Pārvaldīt jauna audio faila augšupielādi
        if ($request->hasFile('audio')) {
            if ($gramata->audio_path) {
                Storage::delete('public/' . $gramata->audio_path);
            }

            $path = $request->file('audio')->store('gramatas/audio', 'public');
            $gramata->audio_path = $path;
        }

        $gramata->save();

        return redirect()->route('gramatas.index')->with('success', 'Grāmata veiksmīgi atjaunināta!');
    }

    /**
     * Grāmatas dzēšana no datubāzes:
     */
    public function destroy(Gramata $gramata)
    {
        // Dzēst grāmatu
        $gramata->delete();
        return redirect()->route('gramatas.index')->with('success', 'Grāmata dzēsta!');
    }

    /**
     * Grāmatas favorīta statusa maiņa:
     */
    public function toggleFavorite(Request $request)
    {
        $userId = Auth::id();
        $bookId = $request->book_id;

        // Pārbaudīt, vai favorīts jau eksistē
        $favorite = UserFavorite::where('user_id', $userId)->where('book_id', $bookId)->first();

        if ($favorite) {
            // Ja favorīts eksistē, dzēš to
            $favorite->delete();
            $isFavorite = false;
        } else {
            // Ja favorīta nav, to izveido
            UserFavorite::create(['user_id' => $userId, 'book_id' => $bookId]);
            $isFavorite = true;
        }

        // Atgriež JSON atbildi ar jauno statusu
        return response()->json([
            'success' => true,
            'favorite' => $isFavorite,
        ]);
    }
}
