<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Gramata;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class GramataApiController extends Controller
{
    /**
     * Get all books.
     */
    public function index()
    {
        return response()->json(Gramata::all(), 200);
    }

    /**
     * Get a single book by ID.
     */
    public function show($id)
    {
        $gramata = Gramata::find($id);

        if (!$gramata) {
            return response()->json(['message' => 'Grāmata nav atrasta!'], 404);
        }

        return response()->json($gramata, 200);
    }

    /**
     * Create a new book.
     */
    public function store(Request $request)
    {
        $request->validate([
            'nosaukums' => 'required|string|max:255',
            'autors' => 'required|string|max:255',
            'audio' => 'nullable|file|mimes:mp3|max:20480', // Max 20MB
        ]);

        // Handle file upload
        $audioPath = $request->hasFile('audio') ? $request->file('audio')->store('audio', 'public') : null;

        // Create a new book
        $gramata = Gramata::create([
            'nosaukums' => $request->nosaukums,
            'autors' => $request->autors,
            'audio_path' => $audioPath,
        ]);

        return response()->json(['message' => 'Grāmata veiksmīgi izveidota!', 'gramata' => $gramata], 201);
    }

    /**
     * Update an existing book.
     */
    public function update(Request $request, $id)
    {
        $gramata = Gramata::find($id);

        if (!$gramata) {
            return response()->json(['message' => 'Grāmata nav atrasta!'], 404);
        }

        $request->validate([
            'nosaukums' => 'required|string|max:255',
            'autors' => 'required|string|max:255',
            'audio' => 'nullable|mimes:mp3|max:20480',
        ]);

        // Handle file update
        if ($request->hasFile('audio')) {
            if ($gramata->audio_path) {
                Storage::delete('public/' . $gramata->audio_path);
            }
            $gramata->audio_path = $request->file('audio')->store('audio', 'public');
        }

        $gramata->update([
            'nosaukums' => $request->nosaukums,
            'autors' => $request->autors,
            'audio_path' => $gramata->audio_path,
        ]);

        return response()->json(['message' => 'Grāmata veiksmīgi atjaunināta!', 'gramata' => $gramata], 200);
    }

    /**
     * Delete a book.
     */
    public function destroy($id)
    {
        $gramata = Gramata::find($id);

        if (!$gramata) {
            return response()->json(['message' => 'Grāmata nav atrasta!'], 404);
        }

        if ($gramata->audio_path) {
            Storage::delete('public/' . $gramata->audio_path);
        }

        $gramata->delete();

        return response()->json(['message' => 'Grāmata veiksmīgi dzēsta!'], 200);
    }
}
