document.addEventListener('DOMContentLoaded', () => {
	
	// Pārbauda, vai ir pieejams user ID (t.i. vai lietotājs ir pieteicies sistēmā)
	const userId = localStorage.getItem('userId');
	if (!userId) {
		alert('Not logged in');
		window.location.href = 'index.html';
		return;
	}
	
	// Tiek atrasti HTML elementi, ar kuriem norisināsies tālākā darbība
	const logoutBtn = document.getElementById('logoutBtn');
	const listEl = document.getElementById('calendar-list');
	const addBtn = document.getElementById('add-btn');
	const addForm = document.getElementById('add-calendar-form');
	const submitCalBtn = document.getElementById('add-calendar-btn');

	// Tiek atrasta forma un modulis
	const modal = document.getElementById('task-modal');
	const form = document.getElementById('task-form');
	const deleteBtn = document.getElementById('delete-task-btn');
	const closeModalBtn = document.getElementById('close-task-modal');

	let calendars = [];
	let active = null;
	let events = [];
	let fc = null;
	let editingId = null;

	init();

	// Funckija, kas parūpējas par formas/moduļa/pogu pareizu darbību
	function init() {
		logoutBtn.onclick = () => {
			localStorage.removeItem('userId');
			window.location.href = 'index.html';
		};
		addBtn.onclick = () => addForm.style.display = addForm.style.display === 'none' ? 'flex' : 'none';
		submitCalBtn.onclick = submitCalendar;
		closeModalBtn.onclick = closeModal;

		form.onsubmit = submitTask;
		deleteBtn.onclick = deleteTask;

		document.addEventListener('click', (e) => {
			if (e.target === modal) closeModal();
		});

		loadCalendars();	// Tiek izsaukta funckija, kas iegūst lietotāja kalendārus no datu bāzes
	}	


	//Funkcija, kas nosūta pieprasījumu uz PHP failu, lai iegūtu kalendārus, kas pieder lietotājam
	async function loadCalendars() {
		const res = await fetch(`php/calendars.php?user_id=${userId}`);
		calendars = await res.json();
		renderList();
		if (calendars.length) setActive(calendars[0]);
	}

	// Funkcija, kas ģenerē lietotāja kalendāru sarakstu
	function renderList() {
		listEl.innerHTML = '';
		for (const cal of calendars) {
			const div = document.createElement('div');
			div.textContent = `${cal.title} (${cal.type})`;
			div.className = 'calendar-item';
			div.style.backgroundColor = cal.color;
			if (active?.id === cal.id) div.classList.add('active');
			div.onclick = () => setActive(cal);
			listEl.append(div);
		}
	}

	// Funckija, kas norāda aktīvo kalendāru
	async function setActive(cal) {
		active = cal;
		renderList();
		await loadTasks();
		drawCalendar();
	}

	// Funkcija, kas, iesūtot aizpildītu kalendāra formu, datus nodod PHP failam
	async function submitCalendar() {
		const title = document.getElementById('calendar-title').value.trim();
		const type = document.getElementById('calendar-type').value;
		const color = document.getElementById('calendar-color').value;
		if (!title || !type) return alert('Fill out fields');

		const res = await fetch('php/calendars.php', {
			method: 'POST',
			headers: { 'Content-Type': 'application/json' },
			body: JSON.stringify({ title, type, color, user_id: userId })
		});
		const cal = await res.json();
		calendars.push(cal);
		renderList();
		setActive(cal);
		addForm.reset();
		addForm.style.display = 'none';
	}

	// Funckija, kas iegūst aktīvā kalendāra darbus
	async function loadTasks() {
		const res = await fetch(`php/tasks.php?calendar_id=${active.id}`);
		const data = await res.json();
		events = data.map(t => ({
			id: t.id,
			title: t.title,
			start: `${t.work_date}T${t.start_time}`,
			end: `${t.work_date}T${t.end_time}`,
			backgroundColor: active.color,
			borderColor: active.color,
			textColor: '#fff',
			extendedProps: { type: t.type }
		}));
	}

	// Funckija, kas ģenerē aktīvo kalendāru
	function drawCalendar() {
		if (fc) fc.destroy();
		fc = new FullCalendar.Calendar(document.getElementById('calendar-container'), {
			initialView: 'dayGridMonth',
			headerToolbar: {
				left: 'prev,next today',
				center: 'title',
				right: 'dayGridMonth,timeGridWeek,timeGridDay'
			},
			events,
			eventClick: info => openModal(info.event),
			dateClick: info => {
				highlightDate(info.dateStr);
				openModal(null, info.dateStr);
			}
		});
		fc.render();
	}

	// Funckija, kas izceļs lietotaja izvēlēto datumu (uz tā ir noklikšķināts) (īsti nedarbojas)
	function highlightDate(dateStr) {
		document.querySelectorAll('.fc-day-selected').forEach(el => el.classList.remove('fc-day-selected'));
		const day = document.querySelector(`.fc-day[data-date="${dateStr}"]`);
		if (day) day.classList.add('fc-day-selected');
	}

	// Funkcija, kas atver moduli
	function openModal(evt = null, date = null) {
		editingId = null;
		form.reset();
		deleteBtn.style.display = 'none';

		if (evt) {
			editingId = evt.id;
			document.getElementById('task-title').value = evt.title;
			document.getElementById('task-date').value = evt.startStr.slice(0, 10);
			document.getElementById('task-start-time').value = evt.startStr.slice(11,16);
			document.getElementById('task-end-time').value = evt.endStr ? evt.endStr.slice(11,16) : '';
			document.getElementById('task-type').value = evt.extendedProps.type;
			deleteBtn.style.display = 'inline-block';
		} else {
			document.getElementById('task-date').value = date;
		}
		modal.style.display = 'block';
	}

	// Funkcija, kas aizver moduli
	function closeModal() {
		editingId = null;
		modal.style.display = 'none';
	}

	// Funckija, kas apstrādā iesniegtā darba moduļa datus un tos nodod PHP failam (nestrādā līdz galam)
	async function submitTask(e) {
		e.preventDefault();
		if (!active || !active.id) return alert('Select calendar');

		const data = new FormData(form);
		data.append('calendar_id', active.id);

		if (editingId) data.append('id', editingId);

		const method = editingId ? 'PUT' : 'POST';
  
		const obj = Object.fromEntries(data.entries());

		const res = await fetch('php/tasks.php', {
			method,
			headers: { 'Content-Type': 'application/json' },
			body: JSON.stringify(obj)
		});
  
		const result = await res.json();
		if(result.error) {
			alert('Error: ' + result.error);
			return;
		}

		closeModal();
		await loadTasks();
		drawCalendar();
	}

	// Funckija, kas apstrādā darbu dzēšanu (nestrādā līdz galam)
	async function deleteTask() {
		if (!editingId) return;
		if (!confirm('Delete?')) return;

		await fetch('php/tasks.php', {
			method: 'DELETE',
			headers: { 'Content-Type': 'application/json' },
			body: JSON.stringify({ id: editingId })
		});

		closeModal();
		await loadTasks();
		drawCalendar();
	}
});