import type { User, Events, Ticket, TicketType } from "@prisma/client";
import type { CartItem } from "@/lib/types"
import { create } from "zustand";
import { persist } from "zustand/middleware";

type AppState = {
    user: User | null
    events: Events[],
    tickets: Ticket[],
    tTypes: TicketType[],
    cart: CartItem[]
    // theme: string
    clientSecret: string | null,
}

export type AppAction = {
    setUser: (newUser: User | null) => void,
    setEvents: (newEvents: Events[]) => void,
    setTickets: (newTickets: Ticket[]) => void,
    setTTypes: (newTTypes: TicketType[]) => void,   
    setCart: (newCart: CartItem[]) => void,
    // setTheme: (newTheme: string) => void,
    setClientSecret: (newClientSecret: string | null) => void
    addToCart: (item: CartItem) => void
    removeFromCart: (itemId: string) => void
    updateCartQuantity: (id: string, qty: number) => void
}

export type AppStore = AppState & AppAction

export const useAppStore = create<AppStore>()(
    persist(
        (set, get) => ({
            user: null,
            events: [],
            tickets: [],
            tTypes: [],
            cart: [],
            theme: 'light',
            clientSecret: null,
            setUser: (newUser) => set({ user: newUser }),
            setEvents: (newEvents) => set({ events: newEvents}),
            setTickets: (newTickets) => set({ tickets: newTickets }),
            setTTypes: (newTTypes) => set({ tTypes: newTTypes }),
            setCart: (newCart) => set({ cart: newCart }),
            // setTheme: (newTheme) => set({ theme: newTheme }),
            setClientSecret: (newClientSecret) => set({ clientSecret: newClientSecret}),
            addToCart: (i) => {
                const cart = get().cart
                const existing = cart.findIndex((is) => is.id === i.id)

                if (existing !== -1){
                    const updatedCart = [...cart]
                    updatedCart[existing].quantity += 1
                    set({ cart: updatedCart })

                }else{
                    set({ cart: [...cart, { ...i, quantity: 1 }] })
                }
            },
            removeFromCart:(id) => {
                set({ cart: get().cart.filter((item) => item.id !== id) })
            },
            updateCartQuantity: (id, qty) => {
                const cart = get().cart
                if (qty <= 0){
                    set({ cart: cart.filter(item => item.id !== id) })
                }else{
                    set({ cart: cart.map(item => item.id === id ? { ...item, quantity: qty } : item)})
                }
                
            }
        }),
        { name: "input-storage" }
    )
)