<?php

namespace App\Controllers;
use Config\Database;
use CodeIgniter\RESTful\ResourceController;

class EntryController extends ResourceController
{
    /* ENTRY FUNCTIONS */

    // Retrieves all entries
    public function GetEntries() {
        $db = Database::connect();
        // Gets all entries and checks if it was retrieved
        $query = $db->query('CALL get_entries()');
        if (empty($query)) {
            return $this->fail(["error" => "Not found!"], 404);
        }
        // Prepares all the entries
        $responses = [];
        foreach ($query->getResult() as $row) {
            $responses[] = [
                "id" => $row->id,
                "title" => $row->title,
            ];
        }
        return $this->respond($responses, 200);
    }

    // Retrieves a certain entry
    public function GetEntry($id = null) {
        $db = Database::connect();
        // Checks if the ID is given
        if (empty($id)) {
            return $this->fail(["error" => "Where data?"], 400);
        }
        // Gets a certain entry and checks if it was retrieved / it exists
        $query = $db->query('CALL get_entry('.$db->escape($id).')');
        if (empty($query)) {
            return $this->fail(["error" => "Not found!"], 404);
        }
        // Prepares data
        $responses = [];
        foreach ($query->getResult() as $row) {
            $responses[] = [
                "id" => $row->id,
                "username" => $row->username,
                "title" => $row->title,
                "description" => $row->description,
                "content" => $row->content,
                "created_at" => $row->created_at,
                "edited_at" => $row->edited_at,
            ];
        }
        return $this->respond($responses, 200);
    }

    // Create a new or updates already existing entry
    public function PostEntry($id = null) {
        $db = Database::connect();
        $item = []; // To get the user's id from session data
        // Checks if the user is logged in. If not, then this function doesn't continue
        $session = service("session");
        if ($session->has("userdata")) {
            $item = $session->get("userdata")->id;
            $session->close();
        }
        else {
            $session->close();
            return $this->fail(["error" => "Not found!"], 404);
        }
        // If the entry's id exists, then the entry will get updated, otherwise created/inserted
        if (!empty($id)) {
            // Checks if the entry exists
            $query = $db->query('CALL get_entry_user_id('.$db->escape($id).')')->getResult()[0];
            if ($query->user_id != $item) {
                return $this->fail(["error" => "Not found!"], 404);
            }
            // Checks if posted data exists
            if (empty($this->request->getPost())) {
                return $this->fail(["error" => "Where data?"], 400);
            }
            // Prepares data for updating
            $content = "";
            $st = $this->request->getPost("stitle");
            $sc = $this->request->getPost("scontent");
            for ($_ = 0; $_ < sizeof($st); $_++) {
                $content .= "<section><h2>".$this->CleanTitle($st[$_])."</h2><div>".$this->CleanContent($sc[$_])."</div></section>";
            }
            $sendData = [
                "title" => $this->CleanTitle($this->request->getPost("title")),
                "description" => $this->CleanContent($this->request->getPost("description")),
                "content" => $content
            ];
            $db->query('CALL put_entry('.$db->escape(json_encode($sendData)).', '.$db->escape($id).')');
            return $this->respond(["message" => "Updated successfully!"], 200);
        }
        else {
            // Checks if posted data exists
            if (empty($this->request->getPost())) {
                return $this->fail(["error" => "Where data?"], 400);
            }
            // Prepares data for insertion
            $sendData = [
                "user_id" => $this->request->getPost("user"),
                "title" => $this->CleanTitle($this->request->getPost("title")),
                "description" => $this->CleanContent($this->request->getPost("description"))
            ];
            $db->query('CALL post_entry('.$db->escape(json_encode($sendData)).')');
            return $this->respond(["message" => "Created successfully!"], 201);
        }
    }

    // Deletes an entry
    public function DeleteEntry($id = null) {
        $db = Database::connect();
        // Checks if the ID is given
        if (empty($id)) {
            return $this->fail(["error" => "Where data?"], 400);
        }
        // Checks if the user is logged in and is the owner of the entry. If not, then the entry doesn't get deleted
        $session = service("session");
        if ($session->has("userdata")) {
            $item = $session->get("userdata")->id;
            $session->close();
            $query = $db->query('CALL get_entry_user_id('.$db->escape($id).')')->getResult()[0];
            if ($query->user_id != $item) {
                return $this->fail(["error" => "Not found!"], 404);
            }
        }
        else {
            $session->close();
            return $this->fail(["error" => "Not found!"], 404);
        }
        // Deletes the entry
        $db->query('CALL delete_entry("'.strval($id).'")');
        return $this->respond(["message" => "Deleted successfully!"], 200);
    }

    // Gets the search results
    public function SearchFunction() {
        $db = Database::connect();
        // Checks if the posted (GET) data exists
        if (empty($this->request->getGet())) {
            return $this->fail(["error" => "Where data?"], 400);
        }
        // Gets the searched value
        $param = urldecode($this->request->getGet("param"));
        // Finds any existing entries that have a similar title to searched value
        $query = $db->query('CALL search_data('.$db->escape($param).')');
        if (empty($query)) {
            return $this->fail(["error" => "Not found!"], 404);
        }
        // Prepares data
        $responses = [];
        foreach ($query->getResult() as $row) {
            $responses[] = [
                "id" => $row->id,
                "username" => $row->username,
                "title" => $row->title,
                "description" => $this->CleanTitle($row->description),
                "created_at" => $row->created_at
            ];
        }
        return $this->respond($responses, 200);
    }

    /* SECTION FUNCTIONS */


    // Gets a specific section from the post
    public function GetSection($id = null, $section = null) {
        $db = Database::connect();
        // Checks if the data is given
        if (empty($id)) {
            return $this->fail(["error" => "Where data?"], 400);
        }
        if (empty($section)) {
            return $this->fail(["error" => "Where data?"], 400);
        }
        // Gets the post
        $query = $db->query('CALL get_entry_section('.$db->escape($id).')');
        if (empty($query)) {
            return $this->fail(["error" => "Not found!"], 404);
        }
        $response = [];
        // Prepares data for return
        foreach ($query->getResult() as $row) {
            $response[] = [
                "id" => $row->id,
                "username" => $row->username,
                "content" => "<section>".explode("<section>", $row->content)[$section],
            ];
        }
        return $this->respond($response, 200);
    }

    // Updates a specific section in a post
    public function PostSection($id = null, $section = null) {
        $db = Database::connect();
        $item = [];
        // Checks if the post belongs to the user who is submitting this request
        $session = service("session");
        if ($session->has("userdata")) {
            $item = $session->get("userdata")->id;
            $session->close();
        }
        else {
            $session->close();
            return $this->fail(["error" => "Not found!"], 404);
        }
        // Gets entry data
        $query = $db->query('CALL get_entry_user_id('.$db->escape($id).')')->getResult()[0];
        if ($query->user_id != $item) {
            return $this->fail(["error" => "Not found!"], 404);
        }
        $query = $db->query('CALL get_entry_section('.$db->escape($id).')');
        if (empty($query)) {
            return $this->fail(["error" => "Not found!"], 404);
        }
        if (empty($this->request->getPost())) {
            return $this->fail(["error" => "Where data?"], 400);
        }
        // Prepares data for return
        $response = [];
        foreach ($query->getResult() as $row) {
            $response[] = [
                "id" => $row->id,
                "content" => $row->content,
            ];
        }
        $st = $this->request->getPost("title");
        $sc = $this->request->getPost("content");
        $content = "";
        // If the section wasn't specified, then the section gets placed at the top of the post
        if ($section > 0) {
            // Inserts the section in a specific place - $section is order in which it will be placed
            for ($_ = 1; $_ < sizeof(explode("<section>", $response[0]["content"])); $_++) {
                if ($_ == $section) {
                    $content .= "<section><h2>".$this->CleanTitle($st)."</h2><div>".$this->CleanContent($sc)."</div></section>";
                }
                else {
                    $content .= "<section>".explode("<section>", $response[0]["content"])[$_];
                }
            }
            $sendData = [
                "content" => $content
            ];
            // Updates the post
            $db->query('CALL update_section('.$db->escape(json_encode($sendData)).', '.$db->escape($id).')');
            return $this->respond(["message" => "Updated successfully!"], 200);
        }
        else {
            // Section gets placed at the top of the post
            $content .= $response[0]["content"]."<section><h2>".$this->CleanTitle($st)."</h2><div>".$this->CleanContent($sc)."</div></section>";
            $sendData = [
                "content" => $content
            ];
            // Updates the post
            $db->query('CALL update_section('.$db->escape(json_encode($sendData)).', '.$db->escape($id).')');
            return $this->respond(["message" => "Added successfully!"], 201);
        }
    }

    // Deletes a section in the post
    public function DeleteSection($id = null, $section = null) {
        $db = Database::connect();
        $item = [];
        // Checks if the post belongs to the user who is submitting this request
        $session = service("session");
        if ($session->has("userdata")) {
            $item = $session->get("userdata")->id;
            $session->close();
        }
        else {
            $session->close();
            return $this->fail(["error" => "Not found!"], 404);
        }
        // Gets entry data
        $query = $db->query('CALL get_entry_user_id('.$db->escape($id).')')->getResult()[0];
        if ($query->user_id != $item) {
            return $this->fail(["error" => "Not found!"], 404);
        }
        $query = $db->query('CALL get_entry_section('.$db->escape($id).')');
        if (empty($query)) {
            return $this->fail(["error" => "Not found!"], 404);
        }
        // Prepares data for return
        $response = [];
        foreach ($query->getResult() as $row) {
            $response[] = [
                "id" => $row->id,
                "content" => $row->content,
            ];
        }
        $st = $this->request->getPost("title");
        $sc = $this->request->getPost("content");
        $content = "";
        // Filters out the section and removes it from the post
        for ($_ = 1; $_ < sizeof(explode("<section>", $response[0]["content"])); $_++) {
            if ($_ == $section) {
                $content .= "";
            }
            else {
                $content .= "<section>".explode("<section>", $response[0]["content"])[$_];
            }
        }
        $sendData = [
            "content" => $content
        ];
        $db->query('CALL update_section('.$db->escape(json_encode($sendData)).', '.$db->escape($id).')');
        return $this->respond(["message" => "Deleted successfully!"], 200);
    }

    /* MISCELLANIOUS FUNCTIONS */

    // Removes restricted HTML tags from content
    public function CleanContent($string = null) {
        // Variables get created
        $allowedTags = array("a", "b", "br", "caption", "i", "ol", "ul", "li", "mark", "p", "pre", "q", "s", "big", "small", "sub", "sup", "table", "td", "th", "tr", "u");
        $cleanedString = ""; // Returned string with allowed HTML tags
        $combination = ""; // Variable for string cleaning
        $foundSomething = false; // Variable for finding HTML tag
        // For loop cleans the string
        for ($_ = 0; $_ < strlen($string); $_++) {
            // Check if it can add the character to the cleaned string
            if ($foundSomething == false) {
                if ($string[$_] != "<") {
                    $cleanedString .= $string[$_];
                    continue;
                }
            }
            // If not, then it goes through the tag checking and cleaning process
            $foundSomething = true;
            $combination .= $string[$_];
            // If this runs, then these aren't tags and don't need to be worried about
            if ($combination == ">" || $combination == "< " || $combination == "<>") {
                $cleanedString .= $combination;
                $combination = "";
                $foundSomething = false;
            }
            // When reaching the end of the HTML tag, it will check, if it's allowed
            if ($string[$_] == ">" && $foundSomething == true) {
                $check = str_replace(">", "", str_replace("<", "", $combination));
                // Process for checking HTML link tag
                if ($check[0] == "a") {
                    $arrayCheck = explode(" ", $check);
                    // If the HTML link tag has only the "a" or is with href, then it's allowed
                    if (count($arrayCheck) == 1) {
                        $cleanedString .= $combination;
                    }
                    else {
                        if (count($arrayCheck) <= 2) {
                            if ($arrayCheck[1][0] == "h") {
                                $cleanedString .= $combination;
                            }
                        }
                    }
                }
                else {
                    // Process for checking any HTML tag
                    $check = str_replace("/", "", $check);
                    if (in_array($check, $allowedTags)) {
                        $cleanedString .= $combination;
                    }
                }
                $combination = "";
                $foundSomething = false;
            }
        }
        // Returns the cleaned string from restricted HTML tags
        return $cleanedString;
    }

    // Removes restricted HTML tags from title
    public function CleanTitle($string = null) {
        // Variables get created
        $cleanedString = ""; // Returned string with allowed HTML tags
        $foundSomething = false; // Variable for finding HTML tag
        // For loop cleans the string
        for ($_ = 0; $_ < strlen($string); $_++) {
            // Check if it can add the character to the cleaned string
            if ($foundSomething == false) {
                if ($string[$_] != "<") {
                    $cleanedString .= $string[$_];
                    continue;
                }
            }
            $foundSomething = true;
            if ($string[$_] == ">") {
                $foundSomething = false;
            }
        }
        // Returns the cleaned string from restricted HTML tags
        return $cleanedString;
    }
}