<?php
session_start();
require_once '../config/database.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

$user_id = $_SESSION['user_id'];
$role = $_SESSION['role'];
$task_id = $_GET['id'] ?? null;

if (!$task_id || !is_numeric($task_id)) {
    die("Invalid task ID.");
}

// Fetch task
$query = ($role === 'admin')
    ? "SELECT * FROM tasks WHERE id = ?"
    : "SELECT * FROM tasks WHERE id = ? AND assigned_to = ?";
$stmt = $conn->prepare($query);
if ($role === 'admin') {
    $stmt->bind_param("i", $task_id);
} else {
    $stmt->bind_param("ii", $task_id, $user_id);
}
$stmt->execute();
$result = $stmt->get_result();
$task = $result->fetch_assoc();

if (!$task) {
    die("Task not found or access denied.");
}

// Handle status update and comments
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!empty($_POST['status']) && $role !== 'admin') {
        $status = trim($_POST['status']);
        $valid_statuses = ['New', 'In Progress', 'Completed'];
        if (in_array($status, $valid_statuses)) {
            $update = $conn->prepare("UPDATE tasks SET status = ? WHERE id = ? AND assigned_to = ?");
            $update->bind_param("sii", $status, $task_id, $user_id);
            $update->execute();
        } else {
            die("Invalid status value.");
        }
    }

    if (!empty($_POST['comment'])) {
        $comment = trim($_POST['comment']);
        $insert = $conn->prepare("INSERT INTO task_comments (task_id, user_id, comment) VALUES (?, ?, ?)");
        $insert->bind_param("iis", $task_id, $user_id, $comment);
        $insert->execute();
    }

    header("Location: view_task.php?id=$task_id");
    exit();
}

// Get comments
$comments = $conn->prepare("SELECT c.comment, c.created_at, u.username 
                            FROM task_comments c 
                            JOIN users u ON c.user_id = u.id 
                            WHERE c.task_id = ? 
                            ORDER BY c.created_at DESC");
$comments->bind_param("i", $task_id);
$comments->execute();
$comment_result = $comments->get_result();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Task Details - TaskPro</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Roboto', sans-serif;
            background-color: #f4f4f9;
            margin: 0;
            padding: 40px 20px;
        }

        .task-container {
            max-width: 700px;
            margin: auto;
            background-color: white;
            border-radius: 12px;
            padding: 30px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }

        h2 {
            margin-top: 0;
            font-size: 28px;
            color: #2c3e50;
        }

        p {
            font-size: 16px;
            color: #333;
            line-height: 1.5;
        }

        .form-section {
            margin-top: 30px;
        }

        label {
            font-weight: 500;
            margin-top: 10px;
            display: block;
            color: #444;
        }

        select, textarea, button {
            width: 100%;
            padding: 10px;
            margin-top: 10px;
            font-size: 15px;
            border: 1px solid #ccc;
            border-radius: 6px;
            box-sizing: border-box;
        }

        button {
            background-color: #2c3e50;
            color: white;
            font-weight: bold;
            cursor: pointer;
            transition: background 0.3s ease;
            border: none;
        }

        button:hover {
            background-color: #1a242f;
        }

        .comments {
            margin-top: 40px;
            background-color: #f8f8fc;
            border-radius: 8px;
            padding: 20px;
        }

        .comments h3 {
            margin-top: 0;
            color: #2c3e50;
        }

        .comment {
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 1px solid #ddd;
        }

        .comment strong {
            color: #000;
        }

        .comment em {
            font-size: 12px;
            color: #777;
            display: block;
        }

        .back-btn {
            display: inline-block;
            margin-top: 30px;
            text-decoration: none;
            background-color: #ccc;
            padding: 10px 18px;
            color: black;
            font-weight: 500;
            border-radius: 6px;
        }

        .back-btn:hover {
            background-color: #bbb;
        }
    </style>
</head>
<body>
<div class="task-container">
    <h2><?php echo htmlspecialchars($task['task_name']); ?></h2>
    <p><strong>Due Date:</strong> <?php echo htmlspecialchars($task['due_date']); ?></p>
    <p><strong>Status:</strong> <?php echo htmlspecialchars($task['status']); ?></p>
    <p><strong>Description:</strong><br><?php echo nl2br(htmlspecialchars($task['description'])); ?></p>

    <?php if ($role !== 'admin'): ?>
        <form method="post" class="form-section">
            <label for="status">Update Status:</label>
            <select name="status" id="status" required>
                <option value="New" <?= $task['status'] === 'New' ? 'selected' : '' ?>>New</option>
                <option value="In Progress" <?= $task['status'] === 'In Progress' ? 'selected' : '' ?>>In Progress</option>
                <option value="Completed" <?= $task['status'] === 'Completed' ? 'selected' : '' ?>>Completed</option>
            </select>
            <button type="submit">Save Status</button>
        </form>

        <form method="post" class="form-section">
            <label for="comment">Add Comment:</label>
            <textarea name="comment" id="comment" rows="4" required></textarea>
            <button type="submit">Post Comment</button>
        </form>
    <?php endif; ?>

    <div class="comments">
        <h3>Comments</h3>
        <?php if ($comment_result->num_rows > 0): ?>
            <?php while ($row = $comment_result->fetch_assoc()): ?>
                <div class="comment">
                    <strong><?php echo htmlspecialchars($row['username']); ?></strong>
                    <em><?php echo $row['created_at']; ?></em>
                    <p><?php echo nl2br(htmlspecialchars($row['comment'])); ?></p>
                </div>
            <?php endwhile; ?>
        <?php else: ?>
            <p>No comments yet.</p>
        <?php endif; ?>
    </div>
    <a href="worker.php" class="back-btn">← Back to Dashboard</a>
</div>
</body>
</html>