document.addEventListener("DOMContentLoaded", function() {
    var header = document.getElementById("header");
    var taskList = document.getElementById("taskList");
    var loginData = JSON.parse(localStorage.getItem("loginData"));

    if (loginData && loginData.username) {
        header.innerHTML = "Welcome, " + loginData.username + " | <a href='#' id='logout'>Logout</a>";

        // Logout functionality
        document.getElementById("logout").addEventListener("click", function(event) {
            event.preventDefault();
            localStorage.removeItem("loginData");
            window.location.href = "login.html"; // Redirect to login page after logout
        });

        document.getElementById("searchInput").addEventListener("input", handleSearch);

        // Display task list
        displayTaskList();
    } else {
        header.innerHTML = "<a href='login.html'>Login</a>";
    }

    document.getElementById("filterSelect").addEventListener("change", displayFilterOptions);

    document.getElementById("sortSelect").addEventListener("change", displayTaskList);

    document.getElementById("specificDate").addEventListener("change", displayTaskList);

    document.getElementById("exportCSV").addEventListener("click", exportToCSV);



    // Initialize categories from localStorage
    initializeCategories();
});

function initializeCategories() {
    var existingCategories = JSON.parse(localStorage.getItem("categories")) || ["work", "personal", "school"];
    updateCategoryDropdown(existingCategories, "categorySelect");
    updateCategoryDropdown(existingCategories, "editCategorySelect");
    updateFilterDropdown(existingCategories);
}

function handleSearch() {
    var searchQuery = document.getElementById("searchInput").value.toLowerCase().trim();
    var savedTasks = JSON.parse(localStorage.getItem("tasks")) || [];

    // Filter tasks based on search query
    var filteredTasks = savedTasks.filter(function(task) {
        return task.task.toLowerCase().includes(searchQuery);
    });

    // Display filtered tasks
    displayFilteredTaskList(filteredTasks);
}

function displayFilteredTaskList(filteredTasks) {
    var taskList = document.getElementById("taskList");
    taskList.innerHTML = ""; // Clear existing task list

    filteredTasks.forEach(function(task) {
        var taskItem = createTaskElement(task);
        taskList.appendChild(taskItem);
    });
}


function updateCategoryDropdown(categories, dropdownId) {
    var dropdown = document.getElementById(dropdownId);
    dropdown.innerHTML = "";
    categories.forEach(function(category) {
        var option = document.createElement("option");
        option.value = category;
        option.textContent = category;
        dropdown.appendChild(option);
    });
}

function updateFilterDropdown(categories) {
    var filterSelect = document.getElementById("filterSelect");
    filterSelect.innerHTML = "";

    var allOption = document.createElement("option");
    allOption.value = "";
    allOption.textContent = "All";
    filterSelect.appendChild(allOption);

    categories.forEach(function(category) {
        var option = document.createElement("option");
        option.value = category;
        option.textContent = category;
        filterSelect.appendChild(option);
    });
}

function createNewCategory() {
    var categoryInput = document.getElementById("newCategory").value.trim();
    if (categoryInput === "") {
        alert("Please finish entering new category.");
        return;
    }

    // Retrieve existing categories from localStorage or elsewhere
    var existingCategories = JSON.parse(localStorage.getItem("categories")) || [];

    // Check if the category already exists
    if (existingCategories.includes(categoryInput)) {
        alert("Category already exists.");
        return;
    }

    // Add the new category to the existing categories array
    existingCategories.push(categoryInput);

    // Save the updated categories back to localStorage
    localStorage.setItem("categories", JSON.stringify(existingCategories));

    // Clear the input field after adding the category
    document.getElementById("newCategory").value = "";

    // Update the category and filter dropdown menus
    updateCategoryDropdown(existingCategories, "categorySelect");
    updateCategoryDropdown(existingCategories, "editCategorySelect");
    updateFilterDropdown(existingCategories);
}

function editCategory() {
    var selectedCategory = document.getElementById("editCategorySelect").value.trim();
    var newCategoryName = document.getElementById("editCategoryInput").value.trim();

    if (newCategoryName === "") {
        alert("Please enter a new category name.");
        return;
    }

    var existingCategories = JSON.parse(localStorage.getItem("categories")) || [];
    if (!existingCategories.includes(selectedCategory)) {
        alert("Selected category does not exist.");
        return;
    }

    // Update the category name in the existing categories array
    var categoryIndex = existingCategories.indexOf(selectedCategory);
    existingCategories[categoryIndex] = newCategoryName;

    // Update the categories in localStorage
    localStorage.setItem("categories", JSON.stringify(existingCategories));

    // Update the tasks to reflect the category name change
    var savedTasks = JSON.parse(localStorage.getItem("tasks")) || [];
    savedTasks = savedTasks.map(function(task) {
        if (task.category === selectedCategory) {
            task.category = newCategoryName;
        }
        return task;
    });
    localStorage.setItem("tasks", JSON.stringify(savedTasks));

    // Update the category and filter dropdown menus
    updateCategoryDropdown(existingCategories, "categorySelect");
    updateCategoryDropdown(existingCategories, "editCategorySelect");
    updateFilterDropdown(existingCategories);

    // Clear the input field after editing the category
    document.getElementById("editCategoryInput").value = "";

    // Refresh the task list to reflect changes
    displayTaskList();
}

function addTask() {
    var taskInput = document.getElementById("taskInput").value.trim();
    if (taskInput === "") {
        alert("Please enter a task.");
        return;
    }
    var categorySelect = document.getElementById("categorySelect");
    var category = categorySelect.options[categorySelect.selectedIndex].value.trim();

    var savedTasks = JSON.parse(localStorage.getItem("tasks")) || [];
    var newTask = {
        id: new Date().getTime(),
        task: taskInput,
        category: category,
        created: new Date().toLocaleString(),
        dueDate: "",
        completed: false
    };
    savedTasks.push(newTask);
    localStorage.setItem("tasks", JSON.stringify(savedTasks));

    displayTaskList(); // Refresh task list after adding a new task
    document.getElementById("taskInput").value = ""; // Clear input field
}

function displayTaskList() {
    var savedTasks = JSON.parse(localStorage.getItem("tasks")) || [];
    var filterCategory = document.getElementById("filterSelect").value.trim();
    var specificDate = document.getElementById("specificDate").value.trim();
    var sortOption = document.getElementById("sortSelect").value.trim();

    // Filter tasks by category
    var filteredTasks = savedTasks.filter(function(task) {
        return filterCategory === "" || task.category === filterCategory;
    });

    // Filter tasks by specific date
    if (specificDate) {
        filteredTasks = filteredTasks.filter(function(task) {
            return task.created.includes(specificDate) || task.dueDate === specificDate || (task.completedDate && task.completedDate.includes(specificDate));
        });
    }

    // Sort tasks based on the selected option
    filteredTasks.sort(function(a, b) {
        if (sortOption === "created") {
            return new Date(a.created) - new Date(b.created);
        } else if (sortOption === "dueDate") {
            return new Date(a.dueDate) - new Date(b.dueDate);
        } else if (sortOption === "completedDate") {
            return new Date(a.completedDate) - new Date(b.completedDate);
        } else {
            return 0;
        }
    });

    var taskList = document.getElementById("taskList");
    taskList.innerHTML = "";
    filteredTasks.forEach(function(task) {
        var taskElement = createTaskElement(task);
        taskList.appendChild(taskElement);
    });
}


function createTaskElement(task) {
    var taskItem = document.createElement("div");
    taskItem.classList.add("task");
    if (task.completed) {
        taskItem.classList.add("completed");
    }

    // Determine the color based on the due date
    var dueDateColor = getDueDateColor(task.dueDate);

    taskItem.innerHTML = `
        <input id="checkbox" class="custom-checkbox" type="checkbox" ${task.completed ? "checked" : ""} onclick="toggleTaskCompleted(${task.id})">
        <div contenteditable="true" oninput="updateTaskContent(this, ${task.id})">${task.task}</div>
        <div>Category: ${task.category}</div>
        <div>Created: ${task.created}</div>
        <div>
            Due Date: 
            <input type="date" value="${task.dueDate}" onchange="updateTaskDate(this.value, ${task.id})">
            <span class="due-date-icon" style="background-color: ${dueDateColor}"></span>
        </div>
        ${task.completed ? `<div>Completed: ${task.completedDate}</div>` : ""}
        <button onclick="deleteTask(${task.id})">Delete</button>
    `;
    return taskItem;
}

function deleteTask(taskId) {
    var savedTasks = JSON.parse(localStorage.getItem("tasks")) || [];
    var updatedTasks = savedTasks.filter(function(task) {
        return task.id !== taskId;
    });
    localStorage.setItem("tasks", JSON.stringify(updatedTasks));

    displayTaskList(); // Refresh task list
}

function updateTaskDate(date, taskId) {
    var savedTasks = JSON.parse(localStorage.getItem("tasks")) || [];
    var updatedTasks = savedTasks.map(function(task) {
        if (task.id === taskId) {
            task.dueDate = date;
        }
        return task;
    });
    localStorage.setItem("tasks", JSON.stringify(updatedTasks));
    displayTaskList(); // Refresh task list to update due date color
}

function updateTaskContent(editableDiv, taskId) {
    var newContent = editableDiv.textContent.trim();
    var savedTasks = JSON.parse(localStorage.getItem("tasks")) || [];
    var updatedTasks = savedTasks.map(function(task) {
        if (task.id === taskId) {
            task.task = newContent;
        }
        return task;
    });
    localStorage.setItem("tasks", JSON.stringify(updatedTasks));
}

function toggleTaskCompleted(taskId) {
    var savedTasks = JSON.parse(localStorage.getItem("tasks")) || [];
    var updatedTasks = savedTasks.map(function(task) {
        if (task.id === taskId) {
            task.completed = !task.completed;
            if (task.completed) {
                task.completedDate = new Date().toLocaleString();
            } else {
                task.completedDate = null; // Clear completed date if task is unchecked
            }
        }
        return task;
    });
    localStorage.setItem("tasks", JSON.stringify(updatedTasks));
    displayTaskList(); // Refresh task list to reflect completion status
}

function displayFilterOptions() {
    var filterSelect = document.getElementById("filterSelect");
    var selectedCategory = filterSelect.value.trim();

    var savedTasks = JSON.parse(localStorage.getItem("tasks")) || [];
    var filteredTasks = savedTasks.filter(function(task) {
        return selectedCategory === "" || task.category === selectedCategory;
    });

    var taskList = document.getElementById("taskList");
    taskList.innerHTML = ""; // Clear existing task list

    filteredTasks.forEach(function(task) {
        var taskItem = createTaskElement(task);
        taskList.appendChild(taskItem);
    });
}

function getDueDateColor(dueDate) {
    if (!dueDate) return 'transparent'; // No due date set

    var currentDate = new Date();
    var dueDateObj = new Date(dueDate);
    var timeDiff = dueDateObj - currentDate;
    var daysDiff = Math.ceil(timeDiff / (1000 * 60 * 60 * 24));

    if (daysDiff < 0) {
        return 'red'; // Due date has passed
    } else if (daysDiff <= 14) {
        return 'yellow'; // 2 weeks or less
    } else if (daysDiff <= 28) {
        return 'green'; // 4 weeks or less
    } else {
        return 'transparent'; // More than 4 weeks
    }
}

function exportToCSV() {
    var savedTasks = JSON.parse(localStorage.getItem("tasks")) || [];

    if (savedTasks.length === 0) {
        alert("No tasks available to export.");
        return;
    }

    var csvContent = "data:text/csv;charset=utf-8,";
    csvContent += "ID,Task,Category,Created,Due Date,Prediction when Completed,Job done,Completed Date\n"; // CSV header

    savedTasks.forEach(function(task) {
        var row = [
            task.id,
            `"${task.task}"`, // Enclose in quotes to handle commas within the task description
            task.category,
            task.created,
            task.dueDate,
            task.completed,
            task.completed ? task.completedDate : ""
        ].join(",");
        csvContent += row + "\n";
    });

    var encodedUri = encodeURI(csvContent);
    var link = document.createElement("a");
    link.setAttribute("href", encodedUri);
    link.setAttribute("download", "tasks.csv");
    document.body.appendChild(link); // Required for Firefox

    link.click();
    document.body.removeChild(link); // Clean up
}
