// Main variables
let clockInterval, stopwatchInterval;
let startTime, elapsedTime = 0;
let lapCount = 0;
let isPaused = false;
let lastLapTime = 0;
let username = '';

// Event listeners
document.addEventListener('DOMContentLoaded', () => {
    document.getElementById('loginBtn').addEventListener('click', handleLogin);
    document.getElementById('logoutBtn').addEventListener('click', handleLogout);
    document.getElementById('startStopwatchBtn').addEventListener('click', startStopwatch);
    document.getElementById('stopBtn').addEventListener('click', stopStopwatch);
    document.getElementById('pauseBtn').addEventListener('click', pauseStopwatch);
    document.getElementById('lapBtn').addEventListener('click', addLap);
    document.getElementById('deleteLapsBtn').addEventListener('click', deleteLaps);
    document.getElementById('applyCustomizationBtn').addEventListener('click', applyCustomization);
    document.getElementById('closeCustomizationBtn').addEventListener('click', closeCustomization);
    document.getElementById('saveCustomizationBtn').addEventListener('click', saveCustomizations);

    // Make clock elements customizable
    document.querySelectorAll('.customizable').forEach(element => {
        element.addEventListener('click', () => showCustomization(element));
    });

    // Start the clock
    startClock();
});

// Login button handler
function handleLogin() {
    username = document.getElementById('username').value.trim();
    if (username === '') {
        alert('Please enter your name.');
        return;
    }
    document.getElementById('userDisplay').innerText = `User: ${username}`;
    document.getElementById('loginPage').classList.add('hidden');
    document.getElementById('mainPage').classList.remove('hidden');
    loadCustomizations();
    startClock();
}

// Logout button handler
function handleLogout() {
    document.getElementById('loginPage').classList.remove('hidden');
    document.getElementById('mainPage').classList.add('hidden');
    username = '';
    document.getElementById('username').value = '';
}

// Update the clock every second
function updateClock() {
    const now = new Date();
    const hours = now.getHours().toString().padStart(2, '0');
    const minutes = now.getMinutes().toString().padStart(2, '0');
    const seconds = now.getSeconds().toString().padStart(2, '0');
    
    document.getElementById('hours').innerText = hours;
    document.getElementById('minutes').innerText = minutes;
    document.getElementById('seconds').innerText = seconds;
}

// Start the clock interval
function startClock() {
    updateClock();
    clockInterval = setInterval(updateClock, 1000);
}

// Start the stopwatch
function startStopwatch() {
    if (!isPaused) {
        startTime = Date.now() - elapsedTime;
        recordLap('Start');
    } else {
        startTime = Date.now() - elapsedTime;
        isPaused = false;
    }
    stopwatchInterval = setInterval(updateStopwatch, 10);

    document.getElementById('startStopwatchBtn').classList.add('hidden');
    document.getElementById('stopwatchControls').classList.remove('hidden');
    document.getElementById('lapsTable').classList.remove('hidden');

    document.getElementById('stopBtn').innerText = 'Stop';
    document.getElementById('pauseBtn').innerText = 'Pause';
    document.getElementById('pauseBtn').classList.remove('hidden');
    document.getElementById('lapBtn').classList.remove('hidden');
    document.getElementById('pauseBtn').removeEventListener('click', resumeStopwatch);
    document.getElementById('pauseBtn').addEventListener('click', pauseStopwatch);
    document.getElementById('deleteLapsBtn').classList.add('hidden');
}

// Update the stopwatch
function updateStopwatch() {
    elapsedTime = Date.now() - startTime;
    const time = new Date(elapsedTime).toISOString().substr(11, 12);
    document.getElementById('stopwatchTime').innerText = time;
}

// Stop the stopwatch
function stopStopwatch() {
    if (document.getElementById('stopBtn').innerText === 'Stop') {
        clearInterval(stopwatchInterval);
        recordLap('Stop');
        document.getElementById('stopBtn').innerText = 'Save';
        document.getElementById('pauseBtn').classList.add('hidden');
        document.getElementById('lapBtn').classList.add('hidden');
        document.getElementById('deleteLapsBtn').classList.remove('hidden');
    } else {
        saveLaps();
        returnToClock();
    }
}

// Pause the stopwatch
function pauseStopwatch() {
    clearInterval(stopwatchInterval);
    isPaused = true;
    document.getElementById('pauseBtn').innerText = 'Resume';
    document.getElementById('pauseBtn').removeEventListener('click', pauseStopwatch);
    document.getElementById('pauseBtn').addEventListener('click', resumeStopwatch);
    recordLap('Pause');
}

// Resume the stopwatch
function resumeStopwatch() {
    startTime = Date.now() - elapsedTime;
    stopwatchInterval = setInterval(updateStopwatch, 10);
    isPaused = false;
    document.getElementById('pauseBtn').innerText = 'Pause';
    document.getElementById('pauseBtn').removeEventListener('click', resumeStopwatch);
    document.getElementById('pauseBtn').addEventListener('click', pauseStopwatch);
    recordLap('Resume');
}

// Reset the stopwatch
function resetStopwatch() {
    elapsedTime = 0;
    lastLapTime = 0;
}

// Add a lap
function addLap() {
    lapCount++;
    const currentLapTime = elapsedTime;
    const interval = new Date(currentLapTime - lastLapTime).toISOString().substr(11, 12);
    lastLapTime = currentLapTime;
    const lapTime = new Date(currentLapTime).toISOString().substr(11, 12);
    const lapDate = new Date().toLocaleDateString();
    const lapRow = document.createElement('tr');
    lapRow.innerHTML = `<td>${lapCount}</td><td>${lapTime}</td><td>${interval}</td><td>${lapDate}</td><td>Lap</td>`;
    document.getElementById('laps').appendChild(lapRow);
}

// Delete all laps and close table view
function deleteLaps() {
    document.getElementById('laps').innerHTML = '';
    lapCount = 0;
    resetStopwatch();
    returnToClock();  // Close table view when delete button is clicked
}

// Save laps to JSON and download
function saveLaps() {
    const lapRows = document.querySelectorAll('#laps tr');
    const laps = Array.from(lapRows).map(row => {
        const cells = row.getElementsByTagName('td');
        return {
            lap: cells[0].innerText,
            time: cells[1].innerText,
            interval: cells[2].innerText,
            date: cells[3].innerText,
            action: cells[4].innerText
        };
    });
    const json = JSON.stringify(laps, null, 2);

    // Download customizations as JSON file
    const blob = new Blob([json], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'laps.json';
    a.click();
    URL.revokeObjectURL(url);

    // Clear laps after saving
    deleteLaps();  // Call deleteLaps to clear the table and reset stopwatch
}

// Return to clock view
function returnToClock() {
    startClock();
    document.getElementById('clock').classList.remove('hidden');
    document.getElementById('startStopwatchBtn').classList.remove('hidden');
    document.getElementById('stopwatchControls').classList.add('hidden');
    document.getElementById('lapsTable').classList.add('hidden');
}

// Record a lap or action
function recordLap(action) {
    if (action !== 'Start') {
        lapCount++;
        const lapTime = new Date(elapsedTime).toISOString().substr(11, 12);
        const lapDate = new Date().toLocaleDateString();
        const lapRow = document.createElement('tr');
        lapRow.innerHTML = `<td>${lapCount}</td><td>${lapTime}</td><td>-</td><td>${lapDate}</td><td>${action}</td>`;
        document.getElementById('laps').appendChild(lapRow);
    }
}

// Apply customization
function applyCustomization() {
    const color = document.getElementById('color').value;
    const font = document.getElementById('font').value;
    const targetId = document.getElementById('customization').dataset.target;
    const targetElement = document.getElementById(targetId);

    if (targetElement) {
        targetElement.style.color = color;
        targetElement.style.fontFamily = font;
    }

    closeCustomization();
}

// Close customization
function closeCustomization() {
    document.getElementById('customization').classList.add('hidden');
}

// Show customization options
function showCustomization(element) {
    const color = window.getComputedStyle(element).color;
    const fontFamily = window.getComputedStyle(element).fontFamily;

    document.getElementById('color').value = rgbToHex(color);
    document.getElementById('font').value = fontFamily;
    const customizationDiv = document.getElementById('customization');
    customizationDiv.classList.remove('hidden');
    customizationDiv.dataset.target = element.id;
}

// Save customizations to local storage
function saveCustomizations() {
    const customizations = { user: username, settings: {} };
    document.querySelectorAll('.customizable').forEach(element => {
        const part = element.getAttribute('data-part');
        customizations.settings[part] = {
            color: element.style.color || '#000000',
            fontFamily: element.style.fontFamily || 'Arial, sans-serif'
        };
    });

    const json = JSON.stringify(customizations, null, 2);
    localStorage.setItem(`customizations_${username}`, json);

    // Download customizations as JSON file
    const blob = new Blob([json], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'customizations.json';
    a.click();
    URL.revokeObjectURL(url);
}

// Load customizations from local storage
function loadCustomizations() {
    const savedCustomizations = localStorage.getItem(`customizations_${username}`);
    if (savedCustomizations) {
        const customizations = JSON.parse(savedCustomizations);
        Object.keys(customizations.settings).forEach(part => {
            const element = document.querySelector(`.customizable[data-part="${part}"]`);
            if (element) {
                element.style.color = customizations.settings[part].color;
                element.style.fontFamily = customizations.settings[part].fontFamily;
            }
        });
    }
}

// Helper function to convert RGB to Hex
function rgbToHex(rgb) {
    const result = rgb.match(/\d+/g);
    return result ? `#${((1 << 24) + (parseInt(result[0]) << 16) + (parseInt(result[1]) << 8) + parseInt(result[2])).toString(16).slice(1).toUpperCase()}` : '#000000';
}
