document.addEventListener('DOMContentLoaded', () => {
    const usernameInput = document.getElementById('username');
    const registerBtn = document.getElementById('register-btn');
    const userInfo = document.getElementById('user-info');
    const clock = document.getElementById('clock');
    const startBtn = document.getElementById('start-btn');
    const pauseBtn = document.getElementById('pause-btn');
    const stopBtn = document.getElementById('stop-btn');
    const saveBtn = document.getElementById('save-btn');
    const clearBtn = document.getElementById('clear-btn');
    const lapsContainer = document.getElementById('laps');

    const colorHours = document.getElementById('color-hours');
    const colorMinutes = document.getElementById('color-minutes');
    const colorSeconds = document.getElementById('color-seconds');
    const fontSelect = document.getElementById('font-select');
    const saveSettingsBtn = document.getElementById('save-settings-btn');

    let currentUser = null;
    let timerInterval = null;
    let startTime = null;
    let totalPauseTime = 0;
    let lastLapTime = null;
    let clockInterval = null;

    let laps = [];

    registerBtn.addEventListener('click', () => {
        const username = usernameInput.value.trim();
        if (username) {
            currentUser = username;
            userInfo.textContent = `Sveicināti, ${currentUser}`;
            document.getElementById('register-container').style.display = 'none';
            document.getElementById('main-container').style.display = 'block';
            showClock(); // Show clock when registering
            loadSettings();
        }
    });

    function showClock() {
        clockInterval = setInterval(updateClock, 1000);
    }

    function updateClock() {
        const now = new Date();
        const hours = String(now.getHours()).padStart(2, '0');
        const minutes = String(now.getMinutes()).padStart(2, '0');
        const seconds = String(now.getSeconds()).padStart(2, '0');
        clock.innerHTML = `
            <span style="color: ${colorHours.value}">${hours}</span>:
            <span style="color: ${colorMinutes.value}">${minutes}</span>:
            <span style="color: ${colorSeconds.value}">${seconds}</span>
        `;
        clock.style.fontFamily = fontSelect.value;
    }

    saveSettingsBtn.addEventListener('click', () => {
        const settings = {
            colorHours: colorHours.value,
            colorMinutes: colorMinutes.value,
            colorSeconds: colorSeconds.value,
            font: fontSelect.value
        };
        const dataStr = "data:text/json;charset=utf-8," + encodeURIComponent(JSON.stringify(settings));
        const downloadAnchorNode = document.createElement('a');
        downloadAnchorNode.setAttribute("href", dataStr);
        downloadAnchorNode.setAttribute("download", `${currentUser}_settings.json`);
        document.body.appendChild(downloadAnchorNode);
        downloadAnchorNode.click();
        downloadAnchorNode.remove();
        localStorage.setItem(`${currentUser}_settings`, JSON.stringify(settings));
    });

    function loadSettings() {
        const settings = JSON.parse(localStorage.getItem(`${currentUser}_settings`));
        if (settings) {
            colorHours.value = settings.colorHours;
            colorMinutes.value = settings.colorMinutes;
            colorSeconds.value = settings.colorSeconds;
            fontSelect.value = settings.font;
        }
    }

    startBtn.addEventListener('click', () => {
    if (!startTime) {
        startTime = new Date();
        showClock(); // Start clock when starting timer
        laps = [];
        lapsContainer.innerHTML = '';
    }
    if (startBtn.textContent === "Sākt") {
        addLap('Sākt', startTime, "00:00:00");
        startTimer();
        toggleButtons(true, true, true, false, false);
        startBtn.textContent = "Aplis";
    } else {
        const lapStartTime = lastLapTime || startTime;
        const now = new Date();
        const elapsedTime = getElapsedTime(lapStartTime, totalPauseTime);
        addLap('Aplis', now, elapsedTime);
        lastLapTime = now;
    }
});

    pauseBtn.addEventListener('click', () => {
    if (pauseBtn.textContent === "Pauze") {
        clearInterval(timerInterval);
        totalPauseTime += new Date() - lastLapTime;
        addLap('Pauze', new Date(), "00:00:00");
        toggleButtons(false, true, false, false, false);
        pauseBtn.textContent = "Atsākt";
    } else {
        const pauseEndTime = new Date();
        const pauseDuration = getElapsedTime(lastLapTime, totalPauseTime);
        lastLapTime = pauseEndTime;
        startTimer();
        addLap('Atsākt', pauseEndTime, pauseDuration);
        toggleButtons(true, true, true, false, false);
        pauseBtn.textContent = "Pauze";
    }
  });

    stopBtn.addEventListener('click', () => {
        if (timerInterval) {
            clearInterval(timerInterval);
            const now = new Date();
            const elapsedTime = getElapsedTime(startTime, totalPauseTime);
            addLap('Stop', now, elapsedTime);
            toggleButtons(false, false, false, true, true);
            startBtn.textContent = "Sākt";
            stopBtn.textContent = "Saglabāt";
        } else {
            saveBtn.click();
        }
    });

    saveBtn.addEventListener('click', () => {
        const data = JSON.stringify(laps, null, 2);
        const blob = new Blob([data], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `${currentUser}_laps.txt`;
        a.click();
        toggleButtons(false, false, false, false, true);
    });

    clearBtn.addEventListener('click', () => {
        lapsContainer.innerHTML = '';
        startTime = null;
        totalPauseTime = 0;
        lastLapTime = null;
        toggleButtons(true, false, false, false, false);
        startBtn.textContent = "Sākt";
        pauseBtn.textContent = "Pauze";
        stopBtn.textContent = "Stop";
    });

    function startTimer() {
        timerInterval = setInterval(() => {
            updateClock();
        }, 1000);
    }

    function getElapsedTime(start, totalPause) {
        const diff = new Date() - start - totalPause;
        const hours = String(Math.floor(diff / 3600000)).padStart(2, '0');
        const minutes = String(Math.floor((diff % 3600000) / 60000)).padStart(2, '0');
        const seconds = String(Math.floor((diff % 60000) / 1000)).padStart(2, '0');
        return `${hours}:${minutes}:${seconds}`;
    }

    function addLap(type, time, interval) {
        const lap = document.createElement('div');
        lap.className = 'lap';
        lap.textContent = `${type} - ${time.toLocaleTimeString()} - ${interval}`;
        lapsContainer.appendChild(lap);
        laps.push({ type, time: time.toLocaleTimeString(), interval });
    }

    function toggleButtons(start, pause, stop, save, clear) {
        startBtn.disabled = !start;
        pauseBtn.disabled = !pause;
        stopBtn.disabled = !stop;
        saveBtn.disabled = !save;
        clearBtn.disabled = !clear;
    }
});
