let tasks = JSON.parse(localStorage.getItem('tasks')) || [];
let categories = JSON.parse(localStorage.getItem('categories')) || [];

document.addEventListener('DOMContentLoaded', () => {
    loadUser();
    loadCategories();
    displayTasks();
});

function registerUser() {
    const username = document.getElementById('username').value;
    if (username) {
        localStorage.setItem('username', username);
        loadUser();
    }
}

function loadUser() {
    const username = localStorage.getItem('username');
    if (username) {
        document.getElementById('user-info').innerText = `Hello, ${username}`;
        document.getElementById('registration').style.display = 'none';
        document.getElementById('planner').style.display = 'block';
    }
}

function loadCategories() {
    const categorySelect = document.getElementById('task-category');
    categorySelect.innerHTML = '';
    categories.forEach(category => {
        const option = document.createElement('option');
        option.value = category;
        option.innerText = category;
        categorySelect.appendChild(option);
    });
}

function showTaskForm() {
    document.getElementById('task-form').style.display = 'block';
}

function hideTaskForm() {
    document.getElementById('task-form').style.display = 'none';
}

function showCategoryForm() {
    document.getElementById('category-form').style.display = 'block';
}

function hideCategoryForm() {
    document.getElementById('category-form').style.display = 'none';
}

function saveTask() {
    const name = document.getElementById('task-name').value;
    const category = document.getElementById('task-category').value;
    const inputDate = document.getElementById('input-date').value;
    const dueDate = document.getElementById('due-date').value;
    const completionDate = document.getElementById('completion-date').value;

    const task = { name, category, inputDate, dueDate, completionDate };
    tasks.push(task);
    localStorage.setItem('tasks', JSON.stringify(tasks));
    displayTasks();
    hideTaskForm();
}

function saveCategory() {
    const categoryName = document.getElementById('category-name').value;
    categories.push(categoryName);
    localStorage.setItem('categories', JSON.stringify(categories));
    loadCategories();
    hideCategoryForm();
}

function displayTasks() {
    const taskList = document.getElementById('task-list');
    taskList.innerHTML = '';
    tasks.forEach((task, index) => {
        const taskDiv = document.createElement('div');
        taskDiv.innerHTML = `
            <h4>${task.name}</h4>
            <p>Category: ${task.category}</p>
            <p>Input Date: ${task.inputDate}</p>
            <p>Due Date: ${task.dueDate}</p>
            <p>Completion Date: ${task.completionDate}</p>
            <button onclick="deleteTask(${index})">Delete</button>
            <button onclick="markComplete(${index})">Mark Complete</button>
        `;
        taskList.appendChild(taskDiv);
    });
}

function deleteTask(index) {
    tasks.splice(index, 1);
    localStorage.setItem('tasks', JSON.stringify(tasks));
    displayTasks();
}

function markComplete(index) {
    tasks[index].completionDate = new Date().toISOString().split('T')[0];
    localStorage.setItem('tasks', JSON.stringify(tasks));
    displayTasks();
}

function applyFilters() {
    const inputDate = document.getElementById('filter-input-date').value;
    const dueDate = document.getElementById('filter-due-date').value;
    const completionDate = document.getElementById('filter-completion-date').value;

    let filteredTasks = tasks;

    if (inputDate) {
        filteredTasks = filteredTasks.filter(task => task.inputDate === inputDate);
    }

    if (dueDate) {
        filteredTasks = filteredTasks.filter(task => task.dueDate === dueDate);
    }

    if (completionDate) {
        filteredTasks = filteredTasks.filter(task => task.completionDate === completionDate);
    }

    displayFilteredTasks(filteredTasks);
}

function resetFilters() {
    displayTasks();
}

function searchTasks() {
    const searchText = document.getElementById('search-text').value.toLowerCase();
    const filteredTasks = tasks.filter(task => task.name.toLowerCase().includes(searchText));
    displayFilteredTasks(filteredTasks);
}

function displayFilteredTasks(filteredTasks) {
    const taskList = document.getElementById('task-list');
    taskList.innerHTML = '';
    filteredTasks.forEach((task, index) => {
        const taskDiv = document.createElement('div');
        taskDiv.innerHTML = `
            <h4>${task.name}</h4>
            <p>Category: ${task.category}</p>
            <p>Input Date: ${task.inputDate}</p>
            <p>Due Date: ${task.dueDate}</p>
            <p>Completion Date: ${task.completionDate}</p>
        `;
        taskList.appendChild(taskDiv);
    });
}

function exportFilteredTasks() {
    const searchText = document.getElementById('search-text').value.toLowerCase();
    const inputDate = document.getElementById('filter-input-date').value;
    const dueDate = document.getElementById('filter-due-date').value;
    const completionDate = document.getElementById('filter-completion-date').value;

    let filteredTasks = tasks;

    if (inputDate) {
        filteredTasks = filteredTasks.filter(task => task.inputDate === inputDate);
    }

    if (dueDate) {
        filteredTasks = filteredTasks.filter(task => task.dueDate === dueDate);
    }

    if (completionDate) {
        filteredTasks = filteredTasks.filter(task => task.completionDate === completionDate);
    }

    if (searchText) {
        filteredTasks = filteredTasks.filter(task => task.name.toLowerCase().includes(searchText));
    }

    const taskData = filteredTasks.map(task => 
        `Name: ${task.name}\nCategory: ${task.category}\nInput Date: ${task.inputDate}\nDue Date: ${task.dueDate}\nCompletion Date: ${task.completionDate}\n\n`
    ).join('');

    const blob = new Blob([taskData], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'filtered_tasks.txt';
    a.click();
    URL.revokeObjectURL(url);
}
