document.addEventListener('DOMContentLoaded', () => {
    const taskForm = document.getElementById('task-form');
    const taskList = document.getElementById('task-list');
    const categoryForm = document.getElementById('category-form');
    const categoryList = document.getElementById('category-list');
    const taskCategorySelect = document.getElementById('task-category');

    let tasks = [];
    let categories = [];

    fetch('tasks.json')
        .then(response => response.json())
        .then(data => {
            tasks = data;
            renderTasks(tasks);
        });

    fetch('categories.json')
        .then(response => response.json())
        .then(data => {
            categories = data;
            renderCategories();
            updateCategorySelect();
        });

    taskForm.addEventListener('submit', event => {
        event.preventDefault();

        const task = {
            id: Date.now(),
            name: document.getElementById('task-name').value,
            category: document.getElementById('task-category').value,
            taskDate: document.getElementById('task-date').value,
            dueDate: document.getElementById('due-date').value,
            completed: false,
            completionDate: null
        };

        tasks.push(task);
        renderTasks(tasks);
        saveTasks();
    });

    categoryForm.addEventListener('submit', event => {
        event.preventDefault();

        const category = document.getElementById('new-category').value;
        categories.push(category);
        renderCategories();
        updateCategorySelect();
        saveCategories();
    });

    function sortTasks() {
        const sortBy = document.getElementById('sort-by').value;
        if (sortBy) {
            tasks.sort((a, b) => {
                const dateA = a[sortBy] ? new Date(a[sortBy]) : new Date();
                const dateB = b[sortBy] ? new Date(b[sortBy]) : new Date();
                return dateA - dateB;
            });
            renderTasks(tasks);
        }
    }

    function filterTasks() {
        const filterBy = document.getElementById('filter-by').value;
        const filterDate = document.getElementById('filter-date').value;

        if (filterBy && filterDate) {
            const filteredTasks = tasks.filter(task => {
                const taskDate = task[filterBy];
                return new Date(taskDate).toISOString().split('T')[0] === filterDate;
            });
            renderTasks(filteredTasks);
        } else {
            renderTasks(tasks);
        }
    }

    function searchTasks() {
        const searchQuery = document.getElementById('search-task').value.toLowerCase();
        const searchedTasks = tasks.filter(task => task.name.toLowerCase().includes(searchQuery));
        renderTasks(searchedTasks);
    }

    function renderTasks(tasksToRender) {
        taskList.innerHTML = '';
        tasksToRender.forEach(task => {
            const taskItem = document.createElement('li');
            taskItem.className = `task-item ${task.completed ? 'completed' : ''}`;
            taskItem.dataset.id = task.id;

            const completionDateHtml = task.completed ? `<div>Reālās izpildes datums: ${task.completionDate}</div>` : '';
            
            taskItem.innerHTML = `
                <div class="task-info">
                    <div>${task.name}</div>
                    <div>${task.category}</div>
                    <div>Ievades datums: ${task.taskDate}</div>
                    <div>Izpildes datums: ${task.dueDate}</div>
                    ${completionDateHtml}
                </div>
                <div class="btn">
                    <button class="delete-button" style="background-color: #35a931;" onclick="completeTask(${task.id})">Atzīmēt par izpildītu</button>
                    </div>
                    <div class="btn">
                    <button class="edit-button" onclick="editTask(${task.id})">Rediģēt</button>
                    </div>
                    <div class="btn">
                    <button class="delete-button" style="background-color: #dc1d1d;" onclick="deleteTask(${task.id})">Izdzēst</button>
                </div>
            `;
            taskList.appendChild(taskItem);
        });
    }

    function renderCategories() {
        categoryList.innerHTML = '';
        categories.forEach(category => {
            const categoryItem = document.createElement('li');
            categoryItem.className = 'task-item';
            categoryItem.dataset.category = category;
            categoryItem.innerHTML = `
                <span class="category-name">${category}</span>
                <div class="btn">
                    <button class="edit-button" onclick="editCategory('${category}')">Rediģēt</button>
                    </div>
                    <div class="btn">
                    <button class="delete-button" style="background-color: #dc1d1d;" onclick="deleteCategory('${category}')">Izdzēst</button>
                </div>
            `;
            categoryList.appendChild(categoryItem);
        });
    }

    function updateCategorySelect() {
        taskCategorySelect.innerHTML = '';
        categories.forEach(category => {
            const option = document.createElement('option');
            option.value = category;
            option.textContent = category;
            taskCategorySelect.appendChild(option);
        });
    }

    function saveTasks() {
        fetch('tasks.json', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(tasks)
        });
    }

    function saveCategories() {
        fetch('categories.json', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(categories)
        });
    }

    window.completeTask = function (id) {
        const task = tasks.find(t => t.id === id);
        task.completed = true;
        task.completionDate = new Date().toISOString().split('T')[0];
        renderTasks(tasks);
        saveTasks();
    };

    window.editTask = function (id) {
        const taskItem = document.querySelector(`.task-item[data-id='${id}']`);
        const task = tasks.find(t => t.id === id);

        if (taskItem.querySelector('.edit-button').textContent === 'Rediģēt') {
            taskItem.innerHTML = `
                <div class="task-info">
                    <input type="text" id="edit-task-name" value="${task.name}">
                    <input type="text" id="edit-task-category" value="${task.category}">
                    <input type="date" id="edit-task-date" value="${task.taskDate}">
                    <input type="date" id="edit-due-date" value="${task.dueDate}">
                </div>
                <div class="btn">
                    <button class="delete-button" style="background-color: #35a931;" onclick="completeTask(${task.id})">Atzīmēt par izpildītu</button>
                    </div>
                    <div class="btn">
                    <button class="edit-button" onclick="editTask(${task.id})">Saglabāt</button>
                    </div>
                    <div class="btn">
                    <button class="delete-button" style="background-color: #dc1d1d;" onclick="deleteTask(${task.id})">Izdzēst</button>
                </div>
            `;
        } else {
            const updatedTask = {
                id: task.id,
                name: taskItem.querySelector('#edit-task-name').value,
                category: taskItem.querySelector('#edit-task-category').value,
                taskDate: taskItem.querySelector('#edit-task-date').value,
                dueDate: taskItem.querySelector('#edit-due-date').value,
                completed: task.completed,
                completionDate: task.completionDate
            };

            const taskIndex = tasks.findIndex(t => t.id === id);
            tasks[taskIndex] = updatedTask;
            renderTasks(tasks);
            saveTasks();
        }
    };

    document.getElementById("export-button").addEventListener("click", function() {
        // Fetch tasks from the task list
        const tasksToExport = Array.from(document.querySelectorAll(".task-item")).map(taskItem => {
            return {
                id: taskItem.dataset.id,
                name: taskItem.querySelector(".task-info > div:nth-child(1)").textContent.trim(),
                category: taskItem.querySelector(".task-info > div:nth-child(2)").textContent.trim(),
                taskDate: taskItem.querySelector(".task-info > div:nth-child(3)").textContent.trim().split(": ")[1],
                dueDate: taskItem.querySelector(".task-info > div:nth-child(4)").textContent.trim().split(": ")[1],
                completed: taskItem.classList.contains("completed"),
                completionDate: taskItem.querySelector(".task-info > div:nth-child(5)").textContent.trim().split(": ")[1]
            };
        });
    
        console.log(tasksToExport); // Log the tasks to check if data extraction is working
    
        // Convert tasks array to JSON string
        const json = JSON.stringify(tasksToExport, null, 2);
    
        // Create a Blob from the JSON string
        const blob = new Blob([json], {type: "application/json"});
    
        // Create a link element to download the JSON file
        const link = document.createElement("a");
        link.href = window.URL.createObjectURL(blob);
        link.download = "tasks.json";
    
        // Trigger click event to download the JSON file
        link.click();
    });    

    window.editCategory = function (category) {
        const categoryItem = document.querySelector(`.task-item[data-category='${category}']`);
        if (categoryItem.querySelector('.edit-button').textContent === 'Rediģēt') {
            categoryItem.innerHTML = `
                <input type="text" id="edit-category-name" value="${category}">
                <div class="btn">
                    <button class="edit-button" onclick="editCategory('${category}')">Saglabāt</button>
                    </div>
                    <div class="btn">
                    <button class="delete-button" style="background-color: #dc1d1d;" onclick="deleteCategory('${category}')">Izdzēst</button>
                </div>
            `;
        } else {
            const updatedCategory = categoryItem.querySelector('#edit-category-name').value;
            const categoryIndex = categories.findIndex(c => c === category);
            categories[categoryIndex] = updatedCategory;
            renderCategories();
            updateCategorySelect();
            saveCategories();
        }
    };

    window.deleteTask = function (id) {
        tasks = tasks.filter(t => t.id !== id);
        renderTasks(tasks);
        saveTasks();
    };

    window.deleteCategory = function (category) {
        categories = categories.filter(c => c !== category);
        renderCategories();
        updateCategorySelect();
        saveCategories();
    };

    document.getElementById('sort-by').addEventListener('change', sortTasks);
    document.getElementById('filter-by').addEventListener('change', filterTasks);
    document.getElementById('filter-date').addEventListener('change', filterTasks);
    document.getElementById('search-task').addEventListener('keyup', searchTasks);

    document.addEventListener('DOMContentLoaded', function() {
        const userName = localStorage.getItem('userName');
        if (userName) {
            document.getElementById('profile').style.display = 'none';
            document.getElementById('user-name').textContent = userName;
            document.getElementById('user').style.display = 'block';
        }
    });

    function adjustHeight(element) {
        element.style.height = 'auto';
        element.style.height = (element.scrollHeight) + 'px';
    }

    function setDefaultDate() {
        const today = new Date();
        const dateInput = document.getElementById('task-date');
        const formattedDate = today.toISOString().split('T')[0];
        dateInput.value = formattedDate;
    }
});

document.addEventListener('DOMContentLoaded', function() {
    const userName = localStorage.getItem('userName');
    if (userName) {
        document.getElementById('profile').style.display = 'none';
        document.getElementById('user-name').textContent = userName;
        document.getElementById('user').style.display = 'block';
    }
});

function adjustHeight(element) {
    element.style.height = 'auto';
    element.style.height = (element.scrollHeight) + 'px';
}

function setDefaultDate() {
    const today = new Date();
    const dateInput = document.getElementById('task-date');
    const formattedDate = today.toISOString().split('T')[0];
    dateInput.value = formattedDate;
}
