let currentYear;
let currentMonth;

document.addEventListener('DOMContentLoaded', function () {
    // Get the current date
    const currentDate = new Date();

    // Initialize current month
    currentYear = currentDate.getFullYear();
    currentMonth = currentDate.getMonth() + 1;

    // Generate the initial calendar
    generateCalendar(currentYear, currentMonth);

    // Add event listeners to next and previous buttons
    document.getElementById('prevBtn').addEventListener('click', function () {
        changeMonth(-1);
    });

    document.getElementById('nextBtn').addEventListener('click', function () {
        changeMonth(1);
    });
});

async function generateCalendar(year, month) {
    const calendarDiv = document.getElementById('calendar');
    let calendarHTML = '';

    calendarHTML += `<h2>${getMonthName(month)} ${year}</h2>`;
    calendarHTML += '<table>';
    calendarHTML += '<tr><th>Sun</th><th>Mon</th><th>Tue</th><th>Wed</th><th>Thu</th><th>Fri</th><th>Sat</th></tr>';

    const daysInMonth = new Date(year, month, 0).getDate();
    const firstDayOfWeek = new Date(year, month - 1, 1).getDay();

    let dayCounter = 1;

    for (let i = 0; i < 6; i++) {
        calendarHTML += '<tr>';

        for (let j = 0; j < 7; j++) {
            if ((i === 0 && j < firstDayOfWeek) || dayCounter > daysInMonth) {
                calendarHTML += '<td></td>';
            } else {
                // Check if there are tasks for the day
                const hasTasks = await checkTasksForDay(year, month, dayCounter);

                // Set the background color based on completion status
                const bgColor = hasTasks ? (await areAllTasksCompleted(year, month, dayCounter) ? 'green' : 'red') : '';

                calendarHTML += `<td onclick="handleDateClick(${year}, ${month}, ${dayCounter})" style="background-color: ${bgColor};">${dayCounter}</td>`;
                dayCounter++;
            }
        }

        calendarHTML += '</tr>';
    }

    calendarHTML += '</table>';
    calendarDiv.innerHTML = calendarHTML;
}

function getMonthName(month) {
    const months = [
        'January', 'February', 'March', 'April',
        'May', 'June', 'July', 'August',
        'September', 'October', 'November', 'December'
    ];

    return months[month - 1];
}

function handleDateClick(year, month, day) {
    const dateStr = `${year}-${month < 10 ? '0' + month : month}-${day < 10 ? '0' + day : day}`;

    // Create a popup element
    const popup = document.createElement('div');
    popup.className = 'popup';
    popup.innerHTML = `
        <div class="popup-content">
            <p id="popupDate">You clicked on ${dateStr}</p>
            <label for="eventInput">Event:</label>
            <input type="text" id="eventInput" placeholder="Enter event...">
            <button class="add-btn" onclick="addEvent('${dateStr}')">Add</button>
            <table id="taskTable">
                <thead>
                    <tr>
                        <th>Task</th>
                        <th>Completed</th>
                        <th>Date</th>
                        <th>Action</th>
                    </tr>
                </thead>
                <tbody></tbody>
            </table>
            <button class="close-btn" onclick="closePopup()">Close</button>
        </div>
    `;
    document.body.appendChild(popup);

    // Fetch tasks for the clicked date and populate the table
    fetchTasks(dateStr);

    // Add animation class for sliding in
    setTimeout(() => {
        popup.classList.add('slide-in');
    }, 10);
}


function fetchTasks(dateStr) {
    const taskTableBody = document.querySelector('#taskTable tbody');
    const popupDateElement = document.getElementById('popupDate');

    // Make an AJAX request to fetch tasks for the specific date
    const xhr = new XMLHttpRequest();
    xhr.open('GET', `getTasks.php?date=${encodeURIComponent(dateStr)}`, true);

    xhr.onreadystatechange = function () {
        if (xhr.readyState === 4 && xhr.status === 200) {
            // Parse the JSON response
            const tasks = JSON.parse(xhr.responseText);

            // Check if all tasks are completed
            const allCompleted = tasks.every(task => task.completed);

            // Set background color based on completion status
            popupDateElement.style.backgroundColor = allCompleted ? 'green' : 'red';

            // Clear the existing table rows
            taskTableBody.innerHTML = '';

            // Populate the table with tasks
            tasks.forEach(task => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${task.task_text}</td>
                    <td><input type="checkbox" ${task.completed ? 'checked' : ''} onclick="updateTaskStatus(${task.id}, this.checked)"></td>
                    <td>${task.task_date}</td>
                    <td><button onclick="deleteTask(${task.id})">Delete</button></td>
                `;
                taskTableBody.appendChild(row);
            });
        }
    };

    // Send the GET request
    xhr.send();
}


function closePopup() {
    const popup = document.querySelector('.popup');

    // Add animation class for sliding out
    popup.classList.remove('slide-in');
    
    // Remove the popup element from the DOM after the animation completes
    setTimeout(() => {
        document.body.removeChild(popup);
    }, 300);
}

function changeMonth(delta) {
    currentMonth += delta;

    if (currentMonth < 1) {
        currentMonth = 12;
        currentYear--;
    } else if (currentMonth > 12) {
        currentMonth = 1;
        currentYear++;
    }

    generateCalendar(currentYear, currentMonth);
}

function addEvent(dateStr) {
    const eventInput = document.getElementById('eventInput');
    const eventText = eventInput.value.trim();

    if (eventText !== '') {
        // Make an AJAX request to the PHP script
        const xhr = new XMLHttpRequest();
        xhr.open('POST', 'addEvent.php', true);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');

        xhr.onreadystatechange = function () {
            if (xhr.readyState === 4 && xhr.status === 200) {
                // Display the response (you can customize this part)
                alert(xhr.responseText);
                
                // Clear the input field
                eventInput.value = '';
				location.reload();
                
                // Close the popup
                closePopup();
            }
        };

        // Send the POST request with the event text and date
        xhr.send(`eventText=${encodeURIComponent(eventText)}&date=${encodeURIComponent(dateStr)}`);
    }
}

function updateTaskStatus(taskId, completed) {
    const xhr = new XMLHttpRequest();
    xhr.open('POST', 'updateTaskStatus.php', true);
    xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');

    xhr.onreadystatechange = function () {
        if (xhr.readyState === 4) {
            if (xhr.status === 200) {
                // Display the response (you can customize this part)
                alert(xhr.responseText);
				location.reload();
                // Update the UI based on the response
                const checkbox = document.querySelector(`#taskTable input[data-task-id="${taskId}"]`);
                if (checkbox) {
                    checkbox.checked = completed;
                }
            } else {
                // Handle error scenarios here
                alert('Error updating task status');
            }
        }
    };

    // Send the POST request with task ID and completion status
    xhr.send(`taskId=${encodeURIComponent(taskId)}&completed=${encodeURIComponent(completed ? 1 : 0)}`);
}

async function checkTasksForDay(year, month, day) {
    const dateStr = `${year}-${month < 10 ? '0' + month : month}-${day < 10 ? '0' + day : day}`;
    const url = `getTasks.php?date=${encodeURIComponent(dateStr)}`;

    try {
        const response = await fetch(url);
        if (response.ok) {
            const tasks = await response.json();

            // Return true if there are tasks, false otherwise
            return tasks.length > 0;
        } else {
            console.error('Failed to fetch tasks:', response.status, response.statusText);
            return false;
        }
    } catch (error) {
        console.error('Error fetching tasks:', error);
        return false;
    }
}
async function areAllTasksCompleted(year, month, day) {
    const dateStr = `${year}-${month < 10 ? '0' + month : month}-${day < 10 ? '0' + day : day}`;
    const url = `getTasks.php?date=${encodeURIComponent(dateStr)}`;

    try {
        const response = await fetch(url);
        if (response.ok) {
            const tasks = await response.json();

            // Check if all tasks are completed
            return tasks.every(task => task.completed);
        } else {
            console.error('Failed to fetch tasks:', response.status, response.statusText);
            return false;
        }
    } catch (error) {
        console.error('Error fetching tasks:', error);
        return false;
    }
}
function deleteTask(taskId) {
    const confirmDelete = confirm("Are you sure you want to delete this task?");

    if (confirmDelete) {
        const xhr = new XMLHttpRequest();
        xhr.open('POST', 'deleteTask.php', true);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');

        xhr.onreadystatechange = function () {
            if (xhr.readyState === 4) {
                if (xhr.status === 200) {
                    // Display the response (you can customize this part)
                    alert(xhr.responseText);

                    // Reload the page after deleting the task
                    location.reload();
                } else {
                    // Handle error scenarios here
                    alert('Error deleting task');
                }
            }
        };

        // Send the POST request with the task ID to delete
        xhr.send(`taskId=${encodeURIComponent(taskId)}`);
    }
}
